/* Copyright 2006-2009, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


package net.whirljack.sdj;

// JAVA UTILITY
import java.util.Date;

// SCRIP DU JOUR
import net.whirljack.sdj.gui.MainWindow;

// LOGGING
import org.apache.log4j.Logger;


/**
 * This class acts as an alarm, updating the main window periodically.
 * 
 * @author jeremyb
 */
public class AlarmThread implements Runnable {
    
    /** Logging. */
    private Logger logger = Logger.getLogger(AlarmThread.class);
    
    /** The next time to trigger the alarm. */
    private Date nextEvent = null;
    
    /** Reference to the main window so we can display it again. */
    private MainWindow myWindow = null;
    
    /** The thread that runs the alarm. */
    private Thread runner = null;
    
    /** Indicates that an alarm is set. */
    private boolean alarmSet = false;
    
    
    /** Creates a new instance of AlarmThread */
    public AlarmThread(MainWindow myWindow) {
	this.myWindow = myWindow;
    }


    /**
     * Set the time of the next alarm event
     *
     * @param nextEvent date of the next time the alarm should fire.
     */
    public void setNextEvent(Date nextEvent) {
	this.nextEvent = nextEvent;
	if (this.runner != null) {
	    try {
		this.alarmSet = false;
		this.runner.interrupt();
		Thread.sleep(1000);
	    } catch (Exception e) {
		logger.info("Interrupted runner to set new alarm.", e);
	    } finally {
		this.runner = null;
	    }

	}
	
	// create a thread and start it
	this.alarmSet = true;
	this.runner = new Thread(this, "AlarmThread@" + this.nextEvent);
	this.runner.setDaemon(true);
	this.runner.start();
    }
    

    /**
     * The main run loop.
     *
     * <p>This loop will sleep until the next scheduled event, then wake up
     * and call the wakeup method of the main window.</p>
     */
    public void run() {
	this.logger.debug("Alarm thread started, next event at " + this.nextEvent);
	while (this.alarmSet) {
	    try {
		Thread.sleep(60000);
		if (this.alarmSet) {
		    if (new Date().after(this.nextEvent)) {
			this.logger.info("Waking up.");
			this.nextEvent = null;
			this.myWindow.wakeup();
		    }
		}
		
	    } catch (Exception e) {
		logger.warn("Interrupted", e);
	    }
	}
	this.logger.info("Thread finished.");
    }


    /**
     * Gets the next scheduled event.
     *
     * @return date of the next scheduled event.
     */
    public Date getNextEvent() {
	return nextEvent;
    }
}
