/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj;

// JAVA UTILITY
import java.util.ArrayList;
import java.util.List;

// SDJ
import net.whirljack.sdj.bo.SDJEntryElement;

// LOGGING
import org.apache.log4j.Logger;

// XML
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Parse a Scrip du Jour XML file, building a List of objects representing
 * the Entry elements in the file.
 *
 * @author jeremyb
 */
public class EntryElementListParseHandler extends DefaultHandler {

    /** Logging. */
    private Logger logger = Logger.getLogger(EntryElementParseHandler.class);

    /** Object representing one entry element. */
    private SDJEntryElement entry;

    /**The list of SDJEntryElement objects. */
    private List<SDJEntryElement> entryList;

    /** Flag indicating that we are currently in the Heading element. */
    private boolean inHeading = false;

    /** Flag indicating that we are currently in the Text element. */
    private boolean inText = false;

    /** Holds character data. */
    private StringBuffer buf = new StringBuffer();


    /**
     * Creates a new instance of DataFileSummaryHandler.
     *
     */
    public EntryElementListParseHandler() {
        this.entryList = new ArrayList<SDJEntryElement>();

        this.entry = new SDJEntryElement();
    }

    /**
     * Handle an end element event.
     * At the end of the "heading" element, we unset the inHeading flag, put
     * the contents of the StringBuffer in the heading attribute of our
     * entry object, and clear the StringBuffer.
     *
     * At the end of the "text" element, we unset the inText flag, put
     * the contents of the StringBuffer in the text attribute of our
     * entry object, and clear the StringBuffer.
     *
     * At the end of the "entry" element, we add our entry object to the
     * entry list, and create a new, empty element object.
     */
    @Override
    public void endElement(String uri, String localName, String qName)
            throws SAXException {

        if (qName.equals("heading")) {
            this.inHeading = false;
            this.entry.setHeading(this.buf.toString());
            this.buf.delete(0, this.buf.length());
            this.logger.debug("inHeading=false");
            this.logger.debug("Set heading: " + this.entry.getHeading());

        } else if (qName.equals("text")) {
            this.inText = false;
            this.entry.setText(this.buf.toString());
            this.buf.delete(0, this.buf.length());
            this.logger.debug("inText=false");
            this.logger.debug("Set text: " + this.entry.getText());

        } else if (qName.equals("entry")) {
            this.entryList.add(this.entry);
            this.entry = new SDJEntryElement();
        }


    }

    /**
     * Handle a start element event.
     *
     * At the start of an entry element, we set the date and read attributes
     * in our Entry object.
     *
     * At the start of a "heading" element, we set the inHeading flag.
     * At the start of a "text" element, we set the inText flag.
     */
    @Override
    public void startElement(
            String uri,
            String localName,
            String qName, org.xml.sax.Attributes attributes)
            throws SAXException {

        if (qName.equals("entry")) {
            this.entry.setDate(attributes.getValue("date"));
            if (new Boolean(attributes.getValue("read"))) {
                this.entry.setRead(true);
            } else {
                this.entry.setRead(false);
            }
        } else if (qName.equals("heading")) {
            this.inHeading = true;
            this.logger.debug("inHeading=true");
        } else if (qName.equals("text")) {
            this.inText = true;
            this.logger.debug("inText=true");
        }
    }


    /**
     * Handle character data.
     *
     * The incoming character data is placed in the StringBuffer buf, if we
     * are currently in either the "heading" or "text" elements.
     */
    @Override
    public void characters(char ch[], int start, int length)
            throws SAXException {
        if (this.inText || this.inHeading) {
            this.logger.debug("Got character data");

            for (int i = start; i < start + length; i++) {
                this.buf.append(ch[i]);
            }

        }
    }


    /**
     * Get the list of entry elements that has been built.
     *
     * @return list of Entry objects.
     */
    public List<SDJEntryElement> getEntryElementList() {
        return this.entryList;
    }
}
