/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj;

// JAVA TEXT
import java.text.SimpleDateFormat;

// JAVA UTILITY
import java.util.Date;

// SDJ
import net.whirljack.sdj.bo.SDJEntryElement;

// LOGGING
import org.apache.log4j.Logger;

// XML
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Parse a Scrip du Jour XML file, locating an element for a specific date.
 *
 * @author jeremyb
 */
public class EntryElementParseHandler extends DefaultHandler {

    /** Logging. */
    private Logger logger = Logger.getLogger(EntryElementParseHandler.class);

    /** Object representing the entry element. */
    private SDJEntryElement entry;

    /** The date we are looking for in YYYYMMDD format. */
    private String targetDate;

    /** Flag indicating that we found the correct element. */
    private boolean foundDate = false;

    /** Flag indicating that we are in the heading. */
    private boolean inHeading = false;

    /** Flag indicating that we are in the text. */
    private boolean inText = false;

    /** Holds character data. */
    private StringBuffer buf = new StringBuffer();

    /** Flag indicating to ignore the year portion of the date. */
    private boolean ignoreYear = false;


    /**
     * Default constructor is private.
     */
    private EntryElementParseHandler() {
    }


    /**
     * Creates a new instance of DataFileSummaryHandler.
     *
     * @param date the date to search for.
     * @param ignoreYear if true, the search will only use the month and day.
     */
    public EntryElementParseHandler(Date date, boolean ignoreYear) {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        this.targetDate = format.format(date);
        if (ignoreYear) {
            this.targetDate = this.targetDate.substring(4);
        }
        this.ignoreYear = ignoreYear;
        this.logger.debug("Searching for date " + targetDate + " ignore year is " + ignoreYear);

        this.entry = new SDJEntryElement();
        this.entry.setHeading("No data found for " + targetDate);
    }


    /**
     * Handle an end element event.
     *
     * If we have not found the correct date yet, nothing happens.
     * If we have found the correct date:
     *    If the end of "heading" has been reached, the inHeading flag is
     *    unset, the entry heading attribute is filled with the contents
     *    of the StringBuffer, and the StringBuffer is cleared.
     *
     *    If the end of "test" has been reached, the inText flag is
     *    unset, the entry text attribute is filled with the contents
     *    of the StringBuffer, and the StringBuffer is cleared.
     *
     *    If the end of "entry" has been reached, we set the foundDate
     *    flag to false.  This prevents the now correct data in the
     *    entry object from being overwritten, and allows us to skip some time
     *    consuming processing while parsing the rest of the XML file.
     */
    @Override
    public void endElement(String uri, String localName, String qName)
            throws SAXException {

        // IF WE ARE IN THE CORRECT ENTRY, SET HEADING AND TEXT  AT THE
        // END OF THE ELEMENTS
        if (this.foundDate) {
            if (qName.equals("heading")) {
                this.inHeading = false;
                this.entry.setHeading(this.buf.toString());
                this.buf.delete(0, this.buf.length());
                this.logger.debug("inHeading=false");
                this.logger.debug("Set heading: " + this.entry.getHeading());

            } else if (qName.equals("text")) {
                this.inText = false;
                this.entry.setText(this.buf.toString());
                this.buf.delete(0, this.buf.length());
                this.logger.debug("inContent=false");
                this.logger.debug("Set text: " + this.entry.getText());

            } else if (qName.equals("entry")) {
                this.foundDate = false;
            }
        }

    }


    /**
     * Handle a start of element event.
     *
     * If the date has been found, and we are in "heading" or "text",
     * the flag is set so that we can gather character data.
     *
     * If the date has not been found, and we are at an "entry" element,
     * the date attribute of the entry element is checked against our target
     * date (taking into consideration the ignoreYear flag).  If the dates
     * match, we set the foundDate flag, set the read flag in the entry object,
     * and set the date in the entry object.
     */
    @Override
    public void startElement(
            String uri,
            String localName,
            String qName, org.xml.sax.Attributes attributes)
            throws SAXException {

        if (this.foundDate) {
            if (qName.equals("heading")) {
                this.inHeading = true;
                this.logger.debug("inHeading=true");
            } else if (qName.equals("text")) {
                this.inText = true;
                this.logger.debug("inContent=true");
            }

        } else {
            if (qName.equals("entry")) {
                String date = attributes.getValue("date");
                if (this.ignoreYear) {
                    date = date.substring(4);
                }
                if (date.equals(this.targetDate)) {
                    this.foundDate = true;

                    this.entry.setDate(attributes.getValue("date"));

                    if (new Boolean(attributes.getValue("read"))) {
                        this.entry.setRead(true);
                    } else {
                        this.entry.setRead(false);
                    }
                    this.logger.debug("Found target date " + targetDate);
                }
            }
        }


    }
    

    /**
     * Handle character data.
     */
    @Override
    public void characters(char ch[], int start, int length)
            throws SAXException {
        if (this.inText || this.inHeading) {
            this.logger.debug("Got character data");

            for (int i = start; i < start + length; i++) {
                this.buf.append(ch[i]);
            }

        }
    }

    /**
     * Get the entry element object.
     *
     * @return entry element object for the requested date.
     */
    public SDJEntryElement getSDJEntryElement() {
        return this.entry;
    }
}
