/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj;

// JAVA I/O
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;

// LOGGING
import org.apache.log4j.Logger;


/**
 * Some handy static to manipulate files and streams.
 *
 * @author jeremyb
 */
public class FileUtil {

    /** Logger. */
    private static Logger logger = Logger.getLogger(FileUtil.class);


    /**
     * Private default constructor.
     * All methods in this class should be public static.
     */
    private FileUtil() {
    }


    /**
     * Close an InputStream.
     *
     * <p>Errors will be logged.</p>
     *
     * @param in the InputStream instance to close.
     */
    public static void close(InputStream in) {
        if (in != null) {
            try {
                in.close();
            } catch (Exception e) {
                logger.warn("ERROR CLOSING INPUT STREAM.", e);
            } finally {
                in = null;
            }
        }
    }


    /**
     * Close an OutputStream.
     *
     *<p>Errors will be logged.</p>
     *
     * @param out the OutputStream to close.
     */
    public static void close(OutputStream out) {
        if (out != null) {
            try {
                out.close();
            } catch (Exception e) {
                logger.warn("ERROR CLOSING OUTPUT STREAM.", e);
            } finally {
                out = null;
            }
        }
    }


    /**
     * Close a Writer.
     *
     * <p>Errors will be logged.</p>
     *
     * @param w the writer to close.
     */
    public static void close(Writer w) {
        if (w != null) {
            try {
                w.close();
            } catch (Exception e) {
                logger.warn("ERROR CLOSING WRITER.", e);
            } finally {
                w = null;
            }
        }
    }


    /**
     * Close a Reader.
     *
     * <p>Errors will be logged.</p>
     *
     * @param r the reader to close.
     */
    public static void close(Reader r) {
        if (r != null) {
            try {
                r.close();
            } catch (Exception e) {
                logger.warn("ERROR CLOSING WRITER.", e);
            } finally {
                r = null;
            }
        }
    }


    /**
     * Copy a text file.
     *
     * Encoding is UTF-8.
     *
     * @param source the source file.
     * @param dest the destination file.
     * @throws IOException if any errors occur during the copy.
     */
    public static void copy(File source, File dest) throws IOException {
        BufferedReader in = null;
        BufferedWriter out = null;
        String line = null;

        try {
            in = new BufferedReader(new InputStreamReader(new FileInputStream(source), "UTF-8"));
            out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(dest), "UTF-8"));

            while ((line = in.readLine()) != null) {
                out.write(line);
                out.newLine();
            }

            out.flush();

        } catch (IOException ioe) {
            throw ioe;

        } finally {
            FileUtil.close(in);
            FileUtil.close(out);
        }
    }


    /**
     * Convert old Daily Reader xml files to the new Script du Jour files.
     *
     * The only difference is that the root element is named "document"
     * rather than "dailyReader".
     * All occurances of "<dailyReader" will be replaced with "<document",
     * and all occurances of "</dailyReader>" will be replaced with "</document>".
     *
     * @param source the old file.
     * @param dest the new file.
     * @throws IOException if any errors occur.
     */
    public static void convertToSDJ(File source, File dest) throws IOException {
        BufferedReader in = null;
        BufferedWriter out = null;
        String line = null;

        try {
            in = new BufferedReader(new InputStreamReader(new FileInputStream(source), "UTF-8"));
            out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(dest), "UTF-8"));

            while ((line = in.readLine()) != null) {
                if (line.indexOf("<dailyReader") != -1) {
                    line = line.replaceFirst("<dailyReader", "<document");
                } else if (line.indexOf("</dailyReader>") != -1) {
                    line = line.replaceFirst("</dailyReader>", "</document>");
                }
                out.write(line);
                out.newLine();
            }

            out.flush();

        } catch (IOException ioe) {
            throw ioe;

        } finally {
            FileUtil.close(in);
            FileUtil.close(out);
        }
    }
}
