/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


package net.whirljack.sdj;

// JAVA I/O
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;

// JAVA UTILITY
import java.util.Properties;

// LOGGING
import org.apache.log4j.Logger;



/**
 * Manage properties for the application.
 * Other classes can set and get persistent properties using this class.
 * This class is implemented as a Singleton.
 *
 * @author jeremyb
 */
public class PropertyManager {
    
    /** Logging. */
    private Logger logger = Logger.getLogger(PropertyManager.class);
    
    /** Reference to this class. */
    private static PropertyManager instance = null;
    
    /** The currently set properties. */
    private Properties props = new Properties();
    
    /** Property for automatic update check. */
    public static final String PROPERTY_CHECK_FOR_UPDATES = "sdj.checkForUpdates";
    
    /** Property for alarm clock mode. */
    public static final String PROPERTY_ALARM_CLOCK_MODE = "sdj.alarmClockMode";
    
    /** Property for alarm time. */
    public static final String PROPERTY_ALARM_CLOCK_TIME = "sdj.alarmClockTime";
    
    /** Property for window hide mode. */
    public static final String PROPERTY_WINDOW_HIDE_MODE = "sdj.windowHideMode";

    /** Property for font size. */
    public static final String PROPERTY_FONT_SIZE = "sdj.font.size";

    /** Property for window width. */
    public static final String PROPERTY_WINDOW_WIDTH = "sdj.window.width";

    /** Property for window height. */
    public static final String PROPERTY_WINDOW_HEIGHT = "sdj.window.height";

    /** Property for window x position. */
    public static final String PROPERTY_WINDOW_X = "sdj.window.x";

    /** Property for window y position. */
    public static final String PROPERTY_WINDOW_Y = "sdj.window.y";

    /** Minimize the window. */
    public static final String MODE_MINIMIZE = "minimize";
    
    /** Hide the window. */
    public static final String MODE_HIDE = "hide";
    
    /** Creates a new instance of PropertyManager */
    private PropertyManager() {
    }
    
    
    /**
     * Get a reference to the only instance of this class.
     *
     * @return reference to the instance of PropertyManager.
     */
    public static PropertyManager getInstance() {
	if (instance == null) {
	    instance = new PropertyManager();
	}
	
	return instance;
    }
    
    
    /**
     * Initialize the properties.
     * This method must be called at least once before attempting to get or set
     * properties.  This ensures that the data file exists and is usable.
     *
     * @throws Exception if the properties cannot be loaded or created, or
     *         if any unexpected error occurs.
     */
    public void init() throws Exception {
	// LOAD CONFIGURATION IF POSSIBLE
	InputStream in = null;
	File configFile = new File(SDJMain.getDataDir(), "scripdujour.properties");
	try {
	    if (! configFile.exists()) {
		SDJMain.getDataDir().mkdirs();
		configFile.createNewFile();
		String logfile = (new File(SDJMain.getDataDir(), "scripdujour.log")).getAbsolutePath();
		props = new Properties();
		// SET THE DEFAULT CHECK FOR UPDATES PROPERTY
		props.setProperty(PROPERTY_CHECK_FOR_UPDATES, "true");
		
		// SET LOGGING PROPERTIES
		props.setProperty("log4j.rootLogger", "DEBUG, FILE");
		props.setProperty("log4j.appender.FILE", "org.apache.log4j.RollingFileAppender");
		props.setProperty("log4j.appender.FILE.Threshold", "DEBUG");
		props.setProperty("log4j.appender.FILE.file", logfile);
		props.setProperty("log4j.appender.FILE.layout", "org.apache.log4j.PatternLayout");
		props.setProperty("log4j.appender.FILE.layout.ConversionPattern", "%5p %c [%t] %d{ISO8601} - %m%n");
		props.setProperty("log4j.appender.FILE.MaxFileSize", "2000KB");
		props.setProperty("log4j.appender.FILE.MaxBackupIndex", "5");
		
		// SAVE THE NEW CONFIGURATION FILE
		this.saveProperties();
	    }
	    
	    in = new FileInputStream(configFile);
	    props.load(in);
	    
	    // CHECK FOR THE ALARM CLOCK MODE PROPERTY
	    // SET TO FALSE IF IT DOES NOT EXIST
	    if (props.getProperty(PROPERTY_ALARM_CLOCK_MODE) == null) {
		props.setProperty(PROPERTY_ALARM_CLOCK_MODE, "false");
		props.setProperty(PROPERTY_ALARM_CLOCK_TIME, "06.00");
		this.saveProperties();
	    }
	    
	    // CHECK FOR WINDOW HIDE PROPERTY
	    // SET TO MINIMIZE IF IT DOES NOT EXIST
	    if (props.getProperty(PROPERTY_WINDOW_HIDE_MODE) == null) {
		props.setProperty(PROPERTY_WINDOW_HIDE_MODE, MODE_MINIMIZE);
		this.saveProperties();
	    }

            // SET THE DEFAULT FONT SIZE IF NEEDED
            if (props.getProperty(PropertyManager.PROPERTY_FONT_SIZE) == null) {
                props.setProperty(PropertyManager.PROPERTY_FONT_SIZE, "12");
                this.saveProperties();
            }

            // SET WINDOW POSITION AND SIZE PROPERTIES
            if (props.getProperty(PROPERTY_WINDOW_HEIGHT) == null) {
                java.awt.Dimension screenSize = java.awt.Toolkit.getDefaultToolkit().getScreenSize();
                props.setProperty(PROPERTY_WINDOW_X, Integer.toString((screenSize.width-600)/2));
                props.setProperty(PROPERTY_WINDOW_Y, Integer.toString((screenSize.height-426)/2));
                props.setProperty(PROPERTY_WINDOW_WIDTH, "600");
                props.setProperty(PROPERTY_WINDOW_HEIGHT, "426");
                this.saveProperties();
            }

	    
	} finally {
	    FileUtil.close(in);
	}
    }
    
    
    
    /**
     * Get the current properties.
     *
     * @return the current application properties.
     */
    public Properties getProperties() {
	return this.props;
    }
    
    
    /**
     * Persist the current settings.
     * Any errors will be logged.
     */
    private void saveProperties() {
	OutputStream out = null;
	
	try {
	    File configFile = new File(SDJMain.getDataDir(), "scripdujour.properties");
	    out = new FileOutputStream(configFile);
	    props.store(out, "Scrip du Jour configuration file	");
	    
	} catch (Exception e) {
	    logger.error("COULD NOT SAVE PROPERTIES.", e);
	} finally {
	    FileUtil.close(out);
	}
    }
    
    
    /**
     * Get the named property as a String.
     *
     * @param name name of the property to return.
     * @return the named property, or null if the property does not exist.
     */
    public String getProperty(String name) {
	return this.props.getProperty(name);
    }
    
    
    /**
     * Get the named property as a boolean.
     * 
     * @param name name of the property to return.
     * @return true if the value of the property is "true" or "yes", otherwise
     *         returns false.
     */
    public boolean getPropertyAsBoolean(String name) {
	boolean retval = false;
	
	String val = this.props.getProperty(name);
	if (val != null) {
	    if (val.equalsIgnoreCase("true") || val.equalsIgnoreCase("yes")) {
		retval = true;
	    }
	}
	
	return retval;
    }


    /**
     * Gets the named property as a float.
     *
     * @param name name of the property to return.
     * @return value of the property as a float. If the property does not exist,
     *         or is not a valid float value, 0f is returned.
     */
    public float getPropertyAsFloat(String name) {
        float ret;

        try {
            ret = Float.valueOf(this.props.getProperty(name));
        } catch (Exception e) {
            ret = 0f;
        }

        return ret;
    }


    /**
     * Gets the named property as an int.
     *
     * @param name name of the property to return.
     * @return value of the property as an int. If the property does not exist,
     *         or is not a valid float value, 0 is returned.
     */
    public int getPropertyAsInt(String name) {
        int ret;

        try {
            ret = Integer.valueOf(this.props.getProperty(name));
        } catch (Exception e) {
            ret = 0;
        }

        return ret;
    }

    
    /**
     * Set the property.
     * The properties will be saved.
     *
     * @param name name of the property.
     * @param value value of the property.
     */
    public void setProperty(String name, String value) {
	this.props.setProperty(name, value);
	this.saveProperties();
    }
}
