/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package net.whirljack.sdj;

// JAVA I/O
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Properties;

// SWING
import javax.swing.JOptionPane;

// LOGGING
import net.whirljack.sdj.gui.MacOSEventHandler;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

// SDJ
import net.whirljack.sdj.gui.MainWindow;
import net.whirljack.sdj.gui.SplashWindow;




/**
 * Scrip du Jour entry point.
 * This is the entry point for the Scrip du Jour application.  This class takes
 * care of initializing logging, initializing the XML parser class, and creating
 * the main window.  A reference to the main window is kept here as a static
 * reference, so other classes can get it by calling SDJMain.getMainWindow().
 *
 *
 * @author jeremyb
 */
public class SDJMain {
    
    /** Version. */
    public static String VERSION = "";
    
    /** Logging. */
    private static Logger logger = Logger.getLogger(SDJMain.class);
    
    /** Reference to the main window. */
    private static MainWindow mainWindow;
    
    /** Where we store settings, etc. */
    private static File dataDir = new File(
	    System.getProperty("user.home")
	    + System.getProperty("file.separator")
	    + ".scripdujour");
    
    
    /**
     * Default constructor is private.
     */
    private SDJMain() {
    }
    
    
    /**
     * Application entry point.
     * No command line arguments are supported.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {
	// SET SYSTEM PROPERTY FOR MAC-STYLE MENU BAR
	if (System.getProperty("mrj.version") != null) {
	    System.setProperty("apple.laf.useScreenMenuBar", "true");
            new MacOSEventHandler();
	}

	// API key for Wordnik dictionary
        System.setProperty("WORDNIK_API_KEY", "5458ce49330219f23e0020790810f29b3818096f5fbbf8560");

	
	// SET VERSION
	BufferedReader in = null;
	try {
	    Properties appProps = new Properties();
	    appProps.load(SDJMain.class.getClassLoader().getResourceAsStream("net/whirljack/sdj/VERSION"));
	    SDJMain.VERSION = appProps.getProperty("app.version");
	    
	} catch (Exception e) {
	    SDJMain.VERSION = "0.0.0";
	} finally {
	    FileUtil.close(in);
	}
	
	SplashWindow.getInstance().setStatus("Reading files...");
	new SDJMain().startup();
    }
    
    
    /**
     * Get a reference to the main window.
     * @return reference to the main window.
     */
    public static MainWindow getMainWindow() {
	return mainWindow;
    }
    
    
    /**
     * Get a reference to the data directory.
     * @return data directory.
     */
    public static File getDataDir() {
	return dataDir;
    }
    
    
    /**
     * Do some startup stuff, then create the main window and show it.
     */
    private void startup() {
	try {
	    this.configure();
	    
	} catch (Exception e) {
	    SplashWindow.getInstance().setVisible(false);
	    StringWriter sw = new StringWriter();
	    e.printStackTrace(new PrintWriter(sw));
	    JOptionPane.showMessageDialog(
		    null,
		    "An error occured while running configuration routines."
		    + System.getProperty("line.separator")
		    + sw.toString()
		    + System.getProperty("line.separator")
		    +"Program will abort.",
		    "Fatal Error",
		    JOptionPane.ERROR_MESSAGE);
	    
	    System.exit(1);
	}
	
	logger.debug("Building main window.");
	SplashWindow.getInstance().setStatus("Building main window...");
	
	mainWindow = new MainWindow();
	mainWindow.prepareDisplay();
	
	// HIDE THE SPLASH SCREEN
	SplashWindow.getInstance().close();

        // ADD A SHUTDOWN HOOK
        Runtime.getRuntime().addShutdownHook(new Thread(new ShutdownHook(mainWindow)));

	// SHOW THE WINDOW
	mainWindow.setVisible(true);
    }
    
    
    /**
     * Initialize log4j logging, make sure the XML parser can be instansiated,
     * and migrate data from older versions.
     *
     * The log file will be located in ~/.scripdujour
     */
    private void configure() throws Exception {
	// MAKE SURE TO INIT THE PROPERTY MANAGER!
	PropertyManager.getInstance().init();
	
	PropertyConfigurator.configure(PropertyManager.getInstance().getProperties());
	logger.debug("Logging configured.");
	
	// MAKE SURE THE XML PARSER CAN BE INSTANSIATED
	if (XMLParser.getInstance() == null) {
	    throw new Exception("Could not initialize XML Parser.");
	}
	logger.debug("XML Parser initialized successfully.");
	
	
	// MAKE SURE THINGS MIGRATE FROM THE OLD .dailyreader DIRECTORY
	// TO THE NEW .scripdujour DIRECTORY
	// ALL WE CARE ABOUT ARE THE XML FILES
	File oldDataDir = new File(
		System.getProperty("user.home")
		+ System.getProperty("file.separator")
		+ ".dailyreader");
	
	if (oldDataDir.exists()) {
	    StringBuffer err = new StringBuffer();
	    
	    // MOVE
	    File[] files = oldDataDir.listFiles();
	    for (File f : files) {
		File newFile = new File(dataDir, f.getName());
		
		// CONVERT XML FILE TO NEW FORMAT
		if (f.getName().toLowerCase().endsWith(".xml")) {
		    // CONVERT THIS XML FILE TO NEW FORMAT
		    try {
			FileUtil.convertToSDJ(f, newFile);
		    } catch (IOException ioe) {
			err.append(f.getAbsolutePath()).append(System.getProperty("line.separator"));
		    }
		    
		}
		
		// DELETE THE FILE
		if (f.delete()) {
		    logger.info("Deleted old file " + f.getAbsolutePath());
		} else {
		    logger.error("ERROR DELETING OLD FILE " + f.getAbsolutePath());
		}
	    }
	    if (oldDataDir.delete()) {
		logger.info("All settings migrated.");
	    } else {
		logger.error("ERROR WHILE ATTEMPTING TO DELETE OLD DIRECTORY "
			+ oldDataDir.getAbsolutePath());
	    }
	    
	    if (err.length() != 0) {
		SplashWindow.getInstance().setVisible(false);
		JOptionPane.showMessageDialog(null,
			"Errors occured while migrating these files:" + System.getProperty("line.separator")
			+ err.toString()
			+ "You may need to reinstall data files.",
			"Data Migration Error",
			JOptionPane.WARNING_MESSAGE);
	    }
	}
    }
}
