/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj;

// JAVA I/O
import java.io.File;
import java.io.FileInputStream;

// JAVA UTILITY
import java.util.Date;
import java.util.List;

// XML
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

// SDJ
import net.whirljack.sdj.bo.SDJEntryElement;
import net.whirljack.sdj.bo.SDJRootElement;

// LOGGING
import org.apache.log4j.Logger;

// XML (SAX)
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

/**
 * XML Parsing class.
 *
 * This class is implemented as a singleton.  It provides methods to parse
 * specific data from an XML file.
 *
 * @author jeremyb
 */
public class XMLParser extends DefaultHandler {

    /** Logging object. */
    private static Logger logger = Logger.getLogger(XMLParser.class);

    /** XML reader. */
    private XMLReader xr;
    
    /** Instance of this class. */
    private static XMLParser instance = null;


    /**
     * Creates a new instance of XMLParser.
     *
     * @throws ParserConfigurationException
     * @throws SAXException
     */
    private XMLParser() throws ParserConfigurationException, SAXException {
        super();

        SAXParserFactory spf = SAXParserFactory.newInstance();
        SAXParser sp = spf.newSAXParser();
        this.xr = sp.getXMLReader();
    }


    /**
     * Get a reference to the only instance of this class.
     * If there is an error creating the XML parser, the error will be logged,
     * and this method will return null.  Therefore, it is a good idea to call
     * this method during program startup, and check for a null return.
     *
     * @return instance of the XMLParser class.
     */
    public static XMLParser getInstance() {
        if (instance == null) {
            try {
                instance = new XMLParser();
            } catch (Exception e) {
                logger.error("COULD NOT CREATE XML PARSER!", e);
                instance = null;
            }
        }

        return instance;
    }


    /**
     * Parse the root element from a Scrip du Jour XML file.
     *
     * This method sets the content handler and delegates the work to the
     * parse method.
     *
     * @param file the file to parse.
     * @return object representing the root element.
     * @throws Exception if any errors occur.
     */
    public SDJRootElement parseRootElement(File file) throws Exception {
        RootElementParseHandler handler = new RootElementParseHandler(file);
        this.xr.setContentHandler(handler);
        this.parse(file);

        return handler.getSDJRootElement();
    }


    /**
     * Parse an entry element for a specific date from a Scrip du Jour file.
     *
     * This method sets the content handler and delegates the work to the
     * parse method.
     *
     * @param file the file to parse.
     * @param date the date to look for.
     * @param ignoreYear if true, only the month and day portion of the date is used.
     * @return object representing the data for the requested date.
     * @throws Exception if any errors occur.
     */
    public SDJEntryElement parseEntryForDate(File file, Date date, boolean ignoreYear) throws Exception {
        EntryElementParseHandler handler = new EntryElementParseHandler(date, ignoreYear);
        this.xr.setContentHandler(handler);
        this.parse(file);

        return handler.getSDJEntryElement();
    }


    /**
     * Parse a Scrip du Jour XML file, returning a list of objects representing
     * all the entry elements in the file.
     *
     * This method sets the content handler and delegates the work to the
     * parse method.
     *
     * @param file the file to parse.
     * @return List of objects representing the entry elements in the file.
     * @throws Exception if any errors occur.
     */
    public List<SDJEntryElement> parseEntryElementsToList(File file) throws Exception {
        EntryElementListParseHandler handler = new EntryElementListParseHandler();
        this.xr.setContentHandler(handler);
        this.parse(file);

        return handler.getEntryElementList();
    }


    /**
     * Do the actual work.
     * Any errors will be logged.
     *
     * @param file the file to parse.
     * @throws Exception if any errors occur.
     */
    private void parse(File file) throws Exception {
        this.xr.setErrorHandler(this);
        InputSource inputSource = new InputSource(new FileInputStream(file));
        this.xr.parse(inputSource);

    }


    /**
     * Handle warning events during parsing.
     *
     * @param sAXParseException
     */
    @Override
    public void warning(SAXParseException sAXParseException) throws SAXException {
        logger.warn("Got a warning during parsing.", sAXParseException);
    }


    /**
     * Handle error events during parsing.
     *
     * @param sAXParseException
     */
    @Override
    public void error(SAXParseException sAXParseException) throws SAXException {
        logger.error("Got an error during parsing.", sAXParseException);
    }


    /**
     * Handle fatal events during parsing.
     *
     * @param sAXParseException
     */
    @Override
    public void fatalError(SAXParseException sAXParseException) throws SAXException {
        logger.fatal("Got a fatal error during parsing.", sAXParseException);
    }
}
