/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package net.whirljack.sdj.bo;

// JAVA I/O
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.ParseException;

// JAVA TEXT
import java.text.SimpleDateFormat;

// JAVA UTILITY
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

// SDJ
import net.whirljack.sdj.XMLParser;
import net.whirljack.sdj.FileUtil;

// LOGGING
import org.apache.log4j.Logger;


/**
 * Encapsulates an XML data file for Scrip du Jour.
 *
 * @author jeremyb
 */
public class SDJDataFile {
    
    /** The XML element for the output file. */
    private static final String XML_ELEMENT = "<?xml version=\"1.0\" encoding=\"utf-8\"?>";
    
    /** Logging. */
    private Logger logger = Logger.getLogger(SDJDataFile.class);
    
    /** Object representing the root element of our file. */
    private SDJRootElement root;
    
    /** List of objects representing entry elements in our file. */
    private List<SDJEntryElement> entryList = new ArrayList<SDJEntryElement>();
    
    /** Map dates to entries. */
    private Map<String, SDJEntryElement> dateMap = new HashMap<String, SDJEntryElement>();
    
    /** The current element we are looking at. */
    private int index = 0;
    
    /** Do date formatting to/from YYYYMMDD format. */
    private SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyyMMdd");
    
    
    /* Require usage of the public constructor. */
    private SDJDataFile() {
    }
    
    
    /**
     * Creates a new instance of SDJDataFile.
     * Use this constructor when you want an empty file to put data into.
     *
     * The file will be created and the data written to file.  However, all the
     * entry's will be empty to begin with.
     *
     * @param year the year that the data is good for.
     * @param file the filename to create.
     * @param description the description attribute of the root element.
     * @param shortDescription the shortDescription attribute of the root element.
     * @param ignoreYear the ignoreYear attribute of the root element.
     * @throws IOException if there is an error writing the file.
     */
    public SDJDataFile(int year, File file, String description, String shortDescription, boolean ignoreYear)
    throws IOException {
	
	// CREATE AND INITIALIZE THE ROOT ELEMENT OBJECT
	this.root = new SDJRootElement();
	this.root.setFile(file);
	this.root.setDescription(description);
	this.root.setShortDescription(shortDescription);
	this.root.setIgnoreYear(ignoreYear);
	
	// CREATE A CALENDAR TO LOOP THROUGH THE DAYS
	Calendar cal = new GregorianCalendar();
	cal.set(GregorianCalendar.DAY_OF_MONTH, 1);
	cal.set(GregorianCalendar.MONTH, 0);
	cal.set(GregorianCalendar.YEAR, year);
		
	// INITIALIZE ENTRY OBJECTS, ONE FOR EACH DAY
	while ((cal.get(GregorianCalendar.YEAR)) == year) {
	    
	    SDJEntryElement entry = new SDJEntryElement();
	    entry.setDate(this.dateFormatter.format(cal.getTime()));
	    this.entryList.add(entry);
	    cal.add(GregorianCalendar.DAY_OF_MONTH, 1);
	}
	
	// CREATE THE FILE
	write();
    }
    
    
    /**
     * Creates a new instance of SDJDataFile.
     * Use this constructor when you want to read the contents of a file
     * that already exists.
     * @param file the file to read data from.
     * @throws Exception if the file does not exist, or if the file cannot
     *         be parsed.
     */
    public SDJDataFile(File file) throws Exception {
	String date = null;
	this.root = XMLParser.getInstance().parseRootElement(file);
	this.entryList = XMLParser.getInstance().parseEntryElementsToList(file);
	for (SDJEntryElement element : this.entryList) {
	    
	    if (this.root.isIgnoreYear()) {
		date = element.getDate().substring(4, 8);
	    } else {
		date = element.getDate();
	    }
	    
	    //System.out.println(date);
	    this.dateMap.put(date, element);
	}
    }
    
    
    /**
     * Move the pointer to the next entry in the list.
     * If the end of the list has been reached, the index will be pointing to
     * the first object in the list.
     */
    public void incrementIndex() {
	this.index++;
	if (this.index == this.entryList.size()) {
	    this.index = 0;
	}
    }
    
    
    /**
     * Move the pointer to the previous entry in the list.
     * When the pointer is moved past the first object in the list, it will
     * automatically wrap around to the last object in the list.
     */
    public void decrementIndex() {
	this.index--;
	if (this.index < 0) {
	    this.index = this.entryList.size() - 1;
	}
    }
    
    
    /**
     * Get the current entry, based on the index.
     * @return currently indexed entry.
     */
    public SDJEntryElement getCurrentEntry() {
	return this.entryList.get(index);
    }
    
    
    /**
     * Get the first unread entry.
     * @return first unread entry, or null if no unread entries exist.
     */
    public SDJEntryElement getFirstUnreadEntry() {
	for (SDJEntryElement element : this.entryList) {
	    if (element.isRead()) {
		// keep looking...
	    } else {
		return element;
	    }
	}

	// DID NOT FIND IT IN THE LIST, RETURN NULL
	return null;
    }
    
    
    /**
     * Set the read flag for a specific date.
     *
     * @param date the date to set the flag for.
     * @param read the value to set the flag to.
     */
    public void setReadFlag(Date date, boolean read) {
	this.setReadFlag(this.dateFormatter.format(date), read);
    }
    
    
    /**
     * Set the read flag for a specific date.
     *
     * @param date the date to set the flag for, in yyyymmdd format.
     * @param read value to set the flag to.
     */
    public void setReadFlag(String date, boolean read) {
	if (this.root.isIgnoreYear()) {
	    date = date.substring(4, 8);
	}
	this.dateMap.get(date).setRead(read);
    }
    
    
    /**
     * Get the number of unread items in this data structure.
     * 
     * @return number of unread items.
     */
    public int getUnreadItemCount() {
	int count = 0;
	Date today = null;
	
	// this is not very elegant, but it ensures that the dates
	// compared are only year, month, and date
	String todayString = dateFormatter.format(new Date());
	try {
	    today = dateFormatter.parse(todayString);
	} catch (ParseException ex) {
		today = new Date();
	}
	
	for (SDJEntryElement element : this.entryList) {
	    if (! element.isRead()) {
		String theDate;
		try {
		    if (this.root.isIgnoreYear()) {
			// MAKE THE YEAR THIS YEAR
			theDate = todayString.substring(0, 4);
			theDate = theDate + element.getDate().substring(4);
		    } else {
			theDate = element.getDate();
		    }
		    Date date = dateFormatter.parse(theDate);
		    if (date.before(today)) {
			count++;
		    }
		} catch (ParseException pe) {
		    logger.warn("Could not parse date " + element.getDate(), pe);
		}
	    }
	}
	
	return count;
    }
    
    
    /**
     * Get the current index.
     * @return current index.
     */
    public int getIndex() {
	return this.index;
    }
    
    
    /**
     * Get the object representing the root element.
     * @return object representing the root element.
     */
    public SDJRootElement getSDJRootElement() {
	return this.root;
    }
    
    
    /**
     * Set the index.
     * If the index is greater than the current size, the index will be set to
     * the last position in the list.
     *
     * @param index the index to move to.
     */
    public void setIndex(int index) {
	if (index > this.entryList.size()) {
	    this.index = entryList.size();
	} else {
	    this.index = index;
	}
    }
    
    
    /** 
     * Get the entry element for a specific date.
     *
     * @param date the date to look up.
     * @return the entry for that date, or null if there is no entry.
     */
    public SDJEntryElement getEntryForDate(Date date) {
	return getEntryForDate(this.dateFormatter.format(date));
    }
    
    
    /**
     * Get the entry element for a specific date.
     *
     * @param date the date to look up in yyyymmdd format.
     * @return entry for the date, or null if there is no entry.
     */
    public SDJEntryElement getEntryForDate(String date) {
	
	// TRIM THE DATE IF NECESSARY
	if (this.root.isIgnoreYear()) {
	    date = date.substring(4, 8);
	}
	
	// LOOK UP THE ENTRY
	return this.dateMap.get(date);
    }
    
    
    /**
     * Write the file.
     * The current contents of this object will be written to the file specified
     * when the object was created.  If the file already exists, it will be
     * replaced.
     *
     * @throws IOException if any errors occur.
     */
    public void write() throws IOException {
	File f = this.root.getFile();
	BufferedWriter out = null;
	logger.debug("File is " + f.getAbsolutePath());
	try {
	    out = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(f), "UTF-8"));
	    
	    // THE FIRST PART OF THE FILE
	    out.write(XML_ELEMENT);
	    out.newLine();
	    out.write("<document description=\"");
	    out.write(this.root.getDescription());
	    out.write("\" shortDescription=\"");
	    out.write(this.root.getShortDescription());
	    out.write("\" ignoreYear=\"");
	    out.write((new Boolean(this.root.isIgnoreYear())).toString());
	    out.write("\">");
	    out.newLine();
	    logger.debug("Ready to write " + this.entryList.size() + " records.");
	    for (SDJEntryElement entry: this.entryList) {
		out.write("<entry date=\"");
		out.write(entry.getDate());
		out.write("\" read=\"");
		if (entry.isRead()) {
		    out.write("true");
		} else {
		    out.write("false");
		}
		out.write("\">");
		out.newLine();
		out.write("<heading>");
		out.write(entry.getHeading());
		out.write("</heading>");
		out.newLine();
		out.write("<text>");
		out.write(entry.getText());
		out.write("</text>");
		out.newLine();
		out.write("</entry>");
		out.newLine();
	    }
	    
	    logger.debug("Wrote entry list successfully.");
	    
	    out.write("</document>");
	    out.newLine();
	    
	    out.flush();
	    
	} catch (Exception e) {
	    logger.error("ERROR WRITING FILE " + f.getAbsolutePath(), e);
	    StringWriter sw = new StringWriter();
	    e.printStackTrace(new PrintWriter(sw));
	    throw new IOException("Error writing file " + f.getAbsolutePath()
	    + sw.toString());
	    
	} finally {
	    FileUtil.close(out);
	}
    }
    
}
