/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj.gui;

// JAVA UTILITY
import java.util.List;

// KNICKER
import net.jeremybrooks.knicker.Knicker;
import net.jeremybrooks.knicker.Knicker.SourceDictionary;
import net.jeremybrooks.knicker.dto.Definition;

// LOGGING
import org.apache.log4j.Logger;


/**
 * Look up the definition of a word, and display it.
 *
 * @author jeremyb
 */
public class DefinitionDialog extends javax.swing.JDialog {

    /** Logging. */
    private Logger logger = Logger.getLogger(DefinitionDialog.class);


    /** Creates new form DefinitionDialog */
    public DefinitionDialog(java.awt.Frame parent, boolean modal) {
        super(parent, modal);
        initComponents();
	logger.info("init complete");
	setBounds((parent.getWidth()/2) + parent.getX() - (400/2),
		(parent.getHeight()/2) + parent.getY() - (322/2),
		400, 322);
	logger.info("bounds set");
    }


    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        jTextPane1 = new javax.swing.JTextPane();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);

        jTextPane1.setContentType("text/html");
        jTextPane1.setEditable(false);
        jScrollPane1.setViewportView(jTextPane1);

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 400, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 300, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents


    /**
     * Define the word.
     *
     * @param word word to define.
     */
    public void define(String word) {
	StringBuilder sb = new StringBuilder("<html><body><h1>");
	sb.append(word).append("</h1><p><i>please wait, looking up definition...</i></p></body></html>");
	this.jTextPane1.setText(sb.toString());

	new Lookup(word).execute();
    }


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTextPane jTextPane1;
    // End of variables declaration//GEN-END:variables



    /**
     * Do the lookup in a separate thread.
     *
     * <p>This is a SwingWorker, and will handle lookup and display.</p>
     *
     */
    class Lookup extends javax.swing.SwingWorker<List<Definition>, Object> {

	/** The word to look up. */
	private String word;

	/** Require use of the non-default constructor */
	private Lookup() {}

	/**
	 * Construct a new instance of Lookup.
	 *
	 * @param word the word to define.
	 */
	public Lookup(String word) {
	    this.word = word;
	}


	/**
	 * Look up the definition of the word.
	 *
	 * <p>This method will try each supported Wordnik dictionary, in this
	 * order:
	 * <ul><li>wordnet</li>
	 * <li>wiktionary</li>
	 * <li>webster</li>
	 * <li>century</li>
	 * </ul></p>
	 *
	 * @return list of definitions for the word.
	 * @throws Exception if there are any errors.
	 */
	@Override
	protected List<Definition> doInBackground() throws Exception {
	    List<Definition> retList = this.getDefintions();

	    if (retList.isEmpty()) {
		// the API is case sensitive, so try a lowecase version
		logger.info("Nothing found; trying again with lowercase version.");
		this.word = this.word.toLowerCase();
		retList = this.getDefintions();
	    }

	    return retList;
	}


	/**
	 * Display the definitions when the lookup is finished.
	 *
	 * <p>If there were no definitions found, display a sane message.</p>
	 */
	@Override
	protected void done() {
	    try {
		List<Definition> list = get();
		if (list == null) {
		    showError("Null value returned. The service may be unavailable. Try again later.");

		} else if (list.isEmpty()) {
		    showError("Could not find definition for \"" + word + "\"");
		    
		} else {
		    StringBuilder sb = new StringBuilder("<html><body><h1>");
		    sb.append(word).append("</h1>");

		    int i = 1;
		    for (Definition def : list) {
			sb.append("<p><b>").append(i++).append(".</b> ");
			sb.append("<i>").append(def.getPartOfSpeech()).append(": </i>");
			sb.append(def.getText());
			if (def.getExampleUses().size() > 0) {
			    sb.append(": <i>");
			    sb.append(def.getExampleUses().get(0));
			    sb.append("</i>");
			}
		    }

		    sb.append("</body></html>");

		    jTextPane1.setText(sb.toString());
		}


	    } catch (Exception e) {
		logger.error("Error looking up word.", e);
		showError(e.toString());
	    }

	}


	/**
	 * Convenience method for displaying an error message in the
	 * definition area.
	 *
	 * @param error the error message to display.
	 */
	private void showError(String error) {
	    StringBuilder sb = new StringBuilder("<html><body><h1>");
	    sb.append(word).append("</h1>");
	    sb.append("<p>Could not get the definition. Error was:</p>");
	    sb.append("<p>").append(error).append("</p>");

	    jTextPane1.setText(sb.toString());
	}


	private List<Definition> getDefintions() {
	    List<Definition> retList = null;
	    try {
		logger.info("Looking in ahd");
		retList = Knicker.definitions(word, SourceDictionary.ahd);

		if (retList.isEmpty()) {
		    logger.info("Looking in wordnet");
		    retList = Knicker.definitions(word, SourceDictionary.wordnet);
		}
		
		if (retList.isEmpty()) {
		    logger.info("Looking in wiktionary");
		    retList = Knicker.definitions(word, SourceDictionary.wiktionary);
		}

		if (retList.isEmpty()) {
		    logger.info("Looking in webster");
		    retList = Knicker.definitions(word, SourceDictionary.webster);
		}

		if (retList.isEmpty()) {
		    logger.info("Looking in century");
		    retList = Knicker.definitions(word, SourceDictionary.century);
		}
	    } catch (Exception e) {
		logger.error("Error looking up word.", e);
		showError(e.toString());
	    }

	    return retList;
	}
    }
}
