/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj.gui;

// JAVA I/O
import java.awt.Cursor;
import java.io.File;
import java.io.IOException;

// JAVA SWING
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

// SDJ
import net.whirljack.sdj.SDJMain;
import net.whirljack.sdj.bo.SDJDataFile;

// LOGGING
import org.apache.log4j.Logger;

/**
 * The window shown when a user starts the editor.
 * This window allows the user to either create a new Scrip du Jour file,
 * or open an existing one.  Once the user has done one of those things,
 * the resulting SDJDataFile object is passed to an instance of
 * EditorWindow, where the user can actually edit the file.
 *
 * If the editor closes this window, we go back to the main window.
 *
 * @author  jeremyb
 */
public class EditorStartWindow extends JFrame {

    /** Logging. */
    private Logger logger = Logger.getLogger(EditorStartWindow.class);

    /**
     * Creates new form EditorStartWindow.
     * The location is set relative to where the main window was.
     */
    public EditorStartWindow() {
        initComponents();
        this.setLocation(SDJMain.getMainWindow().getX(), SDJMain.getMainWindow().getY());
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jPanel1 = new javax.swing.JPanel();
        jLabel2 = new javax.swing.JLabel();
        txtFilename = new javax.swing.JTextField();
        jLabel3 = new javax.swing.JLabel();
        txtDescription = new javax.swing.JTextField();
        jLabel4 = new javax.swing.JLabel();
        txtShortDescription = new javax.swing.JTextField();
        jLabel5 = new javax.swing.JLabel();
        txtYear = new javax.swing.JTextField();
        cbxIgnoreYear = new javax.swing.JCheckBox();
        btnNew = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        btnBrowse = new javax.swing.JButton();
        jPanel3 = new javax.swing.JPanel();
        btnCancel = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle("Scrip du Jour Editor: File Selection");
        setIconImage(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/icon_16.png")).getImage());
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });

        jPanel1.setLayout(new java.awt.GridBagLayout());

        jPanel1.setBorder(new javax.swing.border.TitledBorder("Create New File"));
        jLabel2.setText("Filename (optional)");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(jLabel2, gridBagConstraints);

        txtFilename.setToolTipText("Filename to create.  This is relative to your home directory.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(txtFilename, gridBagConstraints);

        jLabel3.setText("Description");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(jLabel3, gridBagConstraints);

        txtDescription.setToolTipText("The description attribute for the data file.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(txtDescription, gridBagConstraints);

        jLabel4.setText("Short Description");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(jLabel4, gridBagConstraints);

        txtShortDescription.setToolTipText("The short description attribute.  This is shown on the tab.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(txtShortDescription, gridBagConstraints);

        jLabel5.setText("Year");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(jLabel5, gridBagConstraints);

        txtYear.setColumns(6);
        txtYear.setToolTipText("Year that data is valid for (if applicable).");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(txtYear, gridBagConstraints);

        cbxIgnoreYear.setText("Ignore Year");
        cbxIgnoreYear.setToolTipText("If checked, the file is good for any year.");
        cbxIgnoreYear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cbxIgnoreYearActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(cbxIgnoreYear, gridBagConstraints);

        btnNew.setText("Create File");
        btnNew.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnNewActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel1.add(btnNew, gridBagConstraints);

        getContentPane().add(jPanel1, java.awt.BorderLayout.NORTH);

        jPanel2.setLayout(new java.awt.GridBagLayout());

        jPanel2.setBorder(new javax.swing.border.TitledBorder("Edit Existing File"));
        jLabel1.setText("To edit an existing file, locate it by clicking the \"Browse\" button");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel2.add(jLabel1, gridBagConstraints);

        btnBrowse.setText("Browse...");
        btnBrowse.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnBrowseActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        jPanel2.add(btnBrowse, gridBagConstraints);

        getContentPane().add(jPanel2, java.awt.BorderLayout.CENTER);

        jPanel3.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        btnCancel.setText("Cancel");
        btnCancel.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnCancelActionPerformed(evt);
            }
        });

        jPanel3.add(btnCancel);

        getContentPane().add(jPanel3, java.awt.BorderLayout.SOUTH);

        pack();
    }
    // </editor-fold>//GEN-END:initComponents

    /**
     * Respond to user clicks on the "Cancel" button.
     * This does the same thing as the user closing the form.
     */
	private void btnCancelActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnCancelActionPerformed
            closeEditorWindow();
	}//GEN-LAST:event_btnCancelActionPerformed

    /**
     * Allow the user to open an existing Scrip du Jour file.
     * The file will be turned into an instance of SDJDataFile,
     * and passed to an EditorWindow.
     *
     * If the file cannot be parsed, an error is displayed, and the user
     * goes back to this window (EditorStartWindow) to try again.
     *
     * The file chooser defaults to the user home, and filters all files
     * that end with ".xml".  Only one file may be selected.
     */
	private void btnBrowseActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnBrowseActionPerformed

            JFileChooser jfc = new JFileChooser(System.getProperty("user.home"));
            jfc.setMultiSelectionEnabled(false);
            jfc.setFileFilter(new SwingXMLFileFilter());
            jfc.setFileHidingEnabled(false);

            if (jfc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
                File f = jfc.getSelectedFile();

                try {
                    this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    SDJDataFile dataFile = new SDJDataFile(f);

                    openEditorWindow(dataFile);

                } catch (Exception e) {
                    logger.error("ERROR CREATING SDJDataFile OBJECT FROM FILE " + f.getAbsolutePath(), e);

                    JOptionPane.showMessageDialog(this,
                            "Could not parse the selected file.",
                            "Parse Error",
                            JOptionPane.ERROR_MESSAGE);
                } finally {
                    this.setCursor(Cursor.getDefaultCursor());
                }

            }
	}//GEN-LAST:event_btnBrowseActionPerformed

    /**
     * If the user closes this window, dispose of it, and show the
     * main window again.
     */
	private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
            closeEditorWindow();
	}//GEN-LAST:event_formWindowClosing

    /**
     * Make the main window visible, and get rid of the editor window.
     */
    private void closeEditorWindow() {
        SDJMain.getMainWindow().setVisible(true);
        this.setVisible(false);
        this.dispose();
    }

    /**
     * If the user has entered valid data, create a new SDJDataFile
     * object with empty entries and pass it to a new EditorWindow.
     *
     * If any errors occur while creating the new file, display an error message
     * and let the user try again.
     */
	private void btnNewActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnNewActionPerformed
            if (validateEntry()) {

                int year = Integer.parseInt(this.txtYear.getText());
                String filename = this.txtFilename.getText();
                if (!filename.endsWith(".xml")) {
                    filename += ".xml";
                }
                File file = new File(System.getProperty("user.home"), filename);

                try {
                    SDJDataFile dataFile = new SDJDataFile(
                            year,
                            file,
                            this.txtDescription.getText(),
                            this.txtShortDescription.getText(),
                            this.cbxIgnoreYear.isSelected());

                    openEditorWindow(dataFile);


                } catch (IOException ioe) {
                    logger.error("ERROR CREATING FILE " + file.getAbsolutePath(), ioe);
                    JOptionPane.showMessageDialog(this,
                            "There was an error while trying to create the file" + System.getProperty("line.separator") + file.getAbsolutePath(),
                            "File Error",
                            JOptionPane.ERROR_MESSAGE);
                }
            }
	}//GEN-LAST:event_btnNewActionPerformed

    /**
     * Check the data entered by a user for a new file.
     * An error message will be displayed if the data fails validation.
     *
     * @return true if the data is okay, false otherwise.
     */
    private boolean validateEntry() {
        boolean valid = true;
        StringBuffer err = new StringBuffer();
        if (this.txtDescription.getText().trim().length() == 0) {
            err.append("  * You must enter a Description." + System.getProperty("line.separator"));
            valid = false;
        }
        if (this.txtShortDescription.getText().trim().length() == 0) {
            err.append("  * You must enter a Short Description." + System.getProperty("line.separator"));
            valid = false;
        }

        if (!this.cbxIgnoreYear.isSelected()) {
            try {
                Integer.parseInt(this.txtYear.getText());
            } catch (Exception e) {
                err.append("  * You must enter a valid Year." + System.getProperty("line.separator"));
                valid = false;
            }
        }

        if (!valid) {
            JOptionPane.showMessageDialog(this,
                    "Please correct the following errors:" + System.getProperty("line.separator") + err.toString(),
                    "Error",
                    JOptionPane.ERROR_MESSAGE);
        } else {
            // LOOKS OKAY, SO CHECK THE FILENAME
            if (this.txtFilename.getText().trim().length() == 0) {
                this.txtFilename.setText(this.txtShortDescription.getText());
            }
        }

        return valid;
    }

    /**
     * Open an EditorWindow.
     * The dataFile is passed to the EditorWindow, and this window is hidden
     * and disposed of.
     * @param dataFile the file representing the SDJMain XML.
     */
    private void openEditorWindow(SDJDataFile dataFile) {
        new EditorWindow(dataFile).setVisible(true);
        this.setVisible(false);
        this.dispose();
    }

    /**
     * If the user selects "Ignore Year", put a default year in the year box,
     * and disable the year box to prevent the user from changing it.  If the
     * user deselectes "Ignore Year", enable the year text box.
     */
	private void cbxIgnoreYearActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cbxIgnoreYearActionPerformed
            if (this.cbxIgnoreYear.isSelected()) {
                this.txtYear.setText("2010");
                this.txtYear.setEnabled(false);
            } else {
                this.txtYear.setText("");
                this.txtYear.setEnabled(true);
            }
	}//GEN-LAST:event_cbxIgnoreYearActionPerformed
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnBrowse;
    private javax.swing.JButton btnCancel;
    private javax.swing.JButton btnNew;
    private javax.swing.JCheckBox cbxIgnoreYear;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JTextField txtDescription;
    private javax.swing.JTextField txtFilename;
    private javax.swing.JTextField txtShortDescription;
    private javax.swing.JTextField txtYear;
    // End of variables declaration//GEN-END:variables
}
