/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package net.whirljack.sdj.gui;

// JAVA AWT
import java.awt.Cursor;

// JAVA I/O
import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;

// JAVA TEXT
import java.text.SimpleDateFormat;
import java.util.Date;

// JAVA SWING
import javax.swing.JOptionPane;

// SDJ
import net.whirljack.sdj.SDJMain;
import net.whirljack.sdj.bo.SDJDataFile;

// LOGGING
import org.apache.log4j.Logger;


/**
 * This window allows a user to edit the Scrip du Jour XML file.
 * Changes are saved whenever the user naviagtes to another day, or when the
 * user closes this window.
 *
 * When this window is closed, the user is taken back to the main window.
 *
 * @author  jeremyb
 */
public class EditorWindow extends javax.swing.JFrame {
    
    /** Logging. */
    private Logger logger = Logger.getLogger(EditorWindow.class);
    
    /** Format dates to/from YYYYMMDD. */
    private SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyyMMdd");
    
    /** Format dates nicely. */
    private DateFormat prettyDateFormat = DateFormat.getDateInstance(DateFormat.MEDIUM);
    
    /** The object model of our XML file. */
    private SDJDataFile dataFile = null;
    
    
    
    /**
     * Creates new form EditorWindow.
     * The location is set based on the location of the main window, and the
     * display is updated.  This should result in the first element of the
     * data file being shown, ready for editing.
     *
     * @param dataFile the object model to work with.
     */
    public EditorWindow(SDJDataFile dataFile) {
	
	this.dataFile = dataFile;
	initComponents();
	this.setLocation(SDJMain.getMainWindow().getX(), SDJMain.getMainWindow().getY());
	updateDisplay();
	
    }
    
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        cbxIgnoreYear = new javax.swing.JCheckBox();
        txtDescription = new javax.swing.JTextField();
        txtShortDescription = new javax.swing.JTextField();
        lblDate = new javax.swing.JLabel();
        jButton1 = new javax.swing.JButton();
        jButton2 = new javax.swing.JButton();
        jLabel5 = new javax.swing.JLabel();
        txtHeading = new javax.swing.JTextField();
        jLabel6 = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();
        txtText = new javax.swing.JTextArea();
        jLabel3 = new javax.swing.JLabel();
        lblFilename = new javax.swing.JLabel();
        jButton3 = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle(this.dataFile.getSDJRootElement().getFile().getAbsolutePath());
        setIconImage(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/icon_16.png")).getImage());
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("Description");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jLabel1, gridBagConstraints);

        jLabel2.setText("Short Description");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jLabel2, gridBagConstraints);

        cbxIgnoreYear.setText("Ignore Year");
        cbxIgnoreYear.setToolTipText("Ignore the year part of the date when displaying data from this file.");
        cbxIgnoreYear.setSelected(this.dataFile.getSDJRootElement().isIgnoreYear());
        cbxIgnoreYear.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cbxIgnoreYearActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(cbxIgnoreYear, gridBagConstraints);

        txtDescription.setColumns(40);
        txtDescription.setText(this.dataFile.getSDJRootElement().getDescription());
        txtDescription.setToolTipText("A description of the data in this file.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(txtDescription, gridBagConstraints);

        txtShortDescription.setColumns(40);
        txtShortDescription.setText(this.dataFile.getSDJRootElement().getShortDescription());
        txtShortDescription.setToolTipText("The short description is shown in the tab when the file is displayed.");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(txtShortDescription, gridBagConstraints);

        lblDate.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(lblDate, gridBagConstraints);

        jButton1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/back.png"))); // NOI18N
        jButton1.setToolTipText("Go to the previous day.");
        jButton1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton1ActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jButton1, gridBagConstraints);

        jButton2.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/next.png"))); // NOI18N
        jButton2.setToolTipText("Go to the next day.");
        jButton2.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton2ActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jButton2, gridBagConstraints);

        jLabel5.setText("Heading");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jLabel5, gridBagConstraints);

        txtHeading.setColumns(40);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(txtHeading, gridBagConstraints);

        jLabel6.setText("Text");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHEAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jLabel6, gridBagConstraints);

        txtText.setLineWrap(true);
        txtText.setWrapStyleWord(true);
        jScrollPane1.setViewportView(txtText);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 7;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jScrollPane1, gridBagConstraints);

        jLabel3.setText("Filename");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jLabel3, gridBagConstraints);

        lblFilename.setText(this.dataFile.getSDJRootElement().getFile().getAbsolutePath());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(lblFilename, gridBagConstraints);

        jButton3.setText("Done");
        jButton3.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                jButton3ActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 8;
        getContentPane().add(jButton3, gridBagConstraints);

        setBounds(0, 0, 600, 400);
    }// </editor-fold>//GEN-END:initComponents

    
    /**
     * Handle when a user clicks the close button on the window.
     */
	private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
	    this.doClose();
	}//GEN-LAST:event_formWindowClosing


        /**
         * Save, show a dialog, and exit.
         *
         */
        private void doClose() {
            saveEntry();

	    JOptionPane.showMessageDialog(this,
		    "File saved successfully.",
		    "Save",
		    JOptionPane.INFORMATION_MESSAGE);

	    SDJMain.getMainWindow().setVisible(true);
	    this.setVisible(false);
	    this.dispose();
        }

        
	/**
	 * When the user clicks the next button, save changes
	 * and move to the next day.
	 */
	private void jButton2ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton2ActionPerformed
	    saveEntry();
	    dataFile.incrementIndex();
	    updateDisplay();
	}//GEN-LAST:event_jButton2ActionPerformed
	
	
	/**
	 * When the user clicks the back button, save changes and move to the
	 * previous day.
	 */
	private void jButton1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton1ActionPerformed
	    saveEntry();
	    dataFile.decrementIndex();
	    updateDisplay();
	}//GEN-LAST:event_jButton1ActionPerformed

        private void cbxIgnoreYearActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cbxIgnoreYearActionPerformed
            // TODO add your handling code here:
        }//GEN-LAST:event_cbxIgnoreYearActionPerformed

        private void jButton3ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_jButton3ActionPerformed
            this.doClose();
        }//GEN-LAST:event_jButton3ActionPerformed
	
	
	/**
	 * Update the display.
	 * The current date, heading and text are set based on the index of the
	 * object model.  The focus is set to the heading.
	 */
	private void updateDisplay() {
	    StringBuffer buf = new StringBuffer();
	    try {
		Date d = this.dateFormatter.parse(this.dataFile.getCurrentEntry().getDate());
		buf.append(this.prettyDateFormat.format(d));
		buf.append(" (");
		buf.append(this.dataFile.getCurrentEntry().getDate());
		buf.append(')');
	    } catch (ParseException ex) {
		logger.error("Could not parse date.", ex);
		buf.append(this.dataFile.getCurrentEntry().getDate());
	    }
	    
	    this.lblDate.setText(buf.toString());
	    this.txtHeading.setText(this.dataFile.getCurrentEntry().getHeading());
	    this.txtHeading.setCaretPosition(0);
	    this.txtText.setText(this.dataFile.getCurrentEntry().getText());
	    this.txtText.setCaretPosition(0);
	    this.txtHeading.requestFocus();
	    
	}
	
	
	/**
	 * Save changes.
	 * The current value of heading, text, description, short description, and
	 * ignore year are passed to the object model.
	 *
	 * Display an error message if there is any error during the save process.
	 * The actual work is delegated to the object model.
	 */
	private void saveEntry() {
	    
	    if (this.dataFile != null) {
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		
		this.dataFile.getCurrentEntry().setHeading(this.txtHeading.getText());
		this.dataFile.getCurrentEntry().setText(this.txtText.getText());
		
		this.dataFile.getSDJRootElement().setDescription(this.txtDescription.getText());
		this.dataFile.getSDJRootElement().setShortDescription(this.txtShortDescription.getText());
		this.dataFile.getSDJRootElement().setIgnoreYear(this.cbxIgnoreYear.isSelected());
		
		
		try {
		    dataFile.write();
		    
		} catch (IOException ioe) {
		    JOptionPane.showMessageDialog(
			    this,
			    "There was a fatal error while writing the file.",
			    "Error",
			    JOptionPane.ERROR_MESSAGE);
		} finally {
		    setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}
	    }
	}
	
	
	
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JCheckBox cbxIgnoreYear;
    private javax.swing.JButton jButton1;
    private javax.swing.JButton jButton2;
    private javax.swing.JButton jButton3;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JLabel lblDate;
    private javax.swing.JLabel lblFilename;
    private javax.swing.JTextField txtDescription;
    private javax.swing.JTextField txtHeading;
    private javax.swing.JTextField txtShortDescription;
    private javax.swing.JTextArea txtText;
    // End of variables declaration//GEN-END:variables
}
