/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj.gui;

// JAVA AWT
import java.awt.Component;
import java.awt.Cursor;

// JAVA I/O
import java.io.File;
import java.io.IOException;

// JAVA TEXT
import java.text.SimpleDateFormat;

// JAVA UTILITY
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.StringTokenizer;

// JAVA SWING
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import net.whirljack.sdj.BrowserLauncher;
import net.whirljack.sdj.AlarmThread;
import net.whirljack.sdj.bo.SDJEntryElement;

// LOGGING
import org.apache.log4j.Logger;

// SDJ
import net.whirljack.sdj.SDJMain;
import net.whirljack.sdj.FileUtil;
import net.whirljack.sdj.VersionChecker;
import net.whirljack.sdj.PropertyManager;


/**
 * The main window.
 * This window allows the user to view data files, and navigate to different days.
 *
 * @author  jeremyb
 */
public class MainWindow extends javax.swing.JFrame {

    /** Logging. */
    private Logger logger = Logger.getLogger(MainWindow.class);

    /** Convert dates to a nice format. */
    private SimpleDateFormat dateFormatter = new SimpleDateFormat("EEEE, MMMM dd, yyyy");

    /** Convert week start dates. */
    private SimpleDateFormat weekDateFormatter = new SimpleDateFormat("MMMM dd, yyyy");

    /** Format dates to/from YYYYMMDD. */
    private SimpleDateFormat entryDateFormatter = new SimpleDateFormat("yyyyMMdd");

    /** The current date. */
    private Calendar currentDate = new GregorianCalendar();

    /** List of TabPanel objects, one for each Scrip du Jour XML file being displayed. */
    private List<TabPanel> tabList;


    /** Change the date for day, week, or month. */
    enum DateChangeMode {

	DAY, WEEK, MONTH

    };
    /** The timer that will display the window every day. */
    private AlarmThread timer = null;


    /**
     * Creates new form MainWindow.
     * The date is set to today, and the tabs are built.
     */
    public MainWindow() {
	this.currentDate.setTime(new Date());

	this.tabList = TabBuilder.getInstance().createTabs();

	initComponents();

	// restore last window size and position
	setBounds(
		PropertyManager.getInstance().getPropertyAsInt(PropertyManager.PROPERTY_WINDOW_X),
		PropertyManager.getInstance().getPropertyAsInt(PropertyManager.PROPERTY_WINDOW_Y),
		PropertyManager.getInstance().getPropertyAsInt(PropertyManager.PROPERTY_WINDOW_WIDTH),
		PropertyManager.getInstance().getPropertyAsInt(PropertyManager.PROPERTY_WINDOW_HEIGHT));

	// SPECIAL HANDLING FOR MAC
	if (System.getProperty("mrj.version") != null) {
	    // HIDE THE File -> Exit MENU ITEM
	    Component[] items = this.fileMenu.getMenuComponents();
	    for (Component item : items) {
		javax.swing.JMenuItem jmi = (javax.swing.JMenuItem) item;
		if (jmi.getText().equals("Exit")) {
		    fileMenu.remove(item);
		}
	    }
	}

	this.timer = new AlarmThread(this);
    }


    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jToolBar1 = new javax.swing.JToolBar();
        backButton = new javax.swing.JButton();
        homeButton = new javax.swing.JButton();
        nextButton = new javax.swing.JButton();
        updateButton = new javax.swing.JButton();
        tabPane = new javax.swing.JTabbedPane();
        lblDate = new javax.swing.JLabel();
        jMenuBar1 = new javax.swing.JMenuBar();
        fileMenu = new javax.swing.JMenu();
        installMenu = new javax.swing.JMenuItem();
        deleteMenu = new javax.swing.JMenuItem();
        menuItemExit = new javax.swing.JMenuItem();
        goMenu = new javax.swing.JMenu();
        menuPreviousDay = new javax.swing.JMenuItem();
        menuToday = new javax.swing.JMenuItem();
        menuNextDay = new javax.swing.JMenuItem();
        jSeparator1 = new javax.swing.JSeparator();
        menuPreviousWeek = new javax.swing.JMenuItem();
        menuNextWeek = new javax.swing.JMenuItem();
        jSeparator2 = new javax.swing.JSeparator();
        menuPreviousMonth = new javax.swing.JMenuItem();
        menuNextMonth = new javax.swing.JMenuItem();
        jSeparator3 = new javax.swing.JSeparator();
        menuFirstUnread = new javax.swing.JMenuItem();
        toolsMenu = new javax.swing.JMenu();
        toolsMenu.addMenuListener(new ToolsMenuHandler());
        editorMenu = new javax.swing.JMenuItem();
        preferencesMenu = new javax.swing.JMenuItem();
        defineMenu = new javax.swing.JMenuItem();
        helpMenu = new javax.swing.JMenu();
        aboutMenu = new javax.swing.JMenuItem();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        setTitle("Scrip du Jour " + SDJMain.VERSION);
        setIconImage(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/icon_16.png")).getImage());
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        getContentPane().setLayout(new java.awt.GridBagLayout());

        backButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/back.png"))); // NOI18N
        backButton.setToolTipText("Go to the previous day.");
        backButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                backButtonActionPerformed(evt);
            }
        });
        jToolBar1.add(backButton);

        homeButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/home.png"))); // NOI18N
        homeButton.setToolTipText("Go to today.");
        homeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                homeButtonActionPerformed(evt);
            }
        });
        jToolBar1.add(homeButton);

        nextButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/next.png"))); // NOI18N
        nextButton.setToolTipText("Go to the next day.");
        nextButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                nextButtonActionPerformed(evt);
            }
        });
        jToolBar1.add(nextButton);

        updateButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/new.png"))); // NOI18N
        updateButton.setText("Update available!");
        updateButton.setToolTipText("A new version is available!  Click here for information.");
        updateButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                updateButtonActionPerformed(evt);
            }
        });
        jToolBar1.add(updateButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(jToolBar1, gridBagConstraints);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(tabPane, gridBagConstraints);

        lblDate.setText("Date");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        getContentPane().add(lblDate, gridBagConstraints);

        fileMenu.setMnemonic('f');
        fileMenu.setText("File");

        installMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/add.png"))); // NOI18N
        installMenu.setMnemonic('i');
        installMenu.setText("Install Data Files");
        installMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                installMenuActionPerformed(evt);
            }
        });
        fileMenu.add(installMenu);

        deleteMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/remove.png"))); // NOI18N
        deleteMenu.setMnemonic('d');
        deleteMenu.setText("Delete Data Files");
        deleteMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                deleteMenuActionPerformed(evt);
            }
        });
        fileMenu.add(deleteMenu);

        menuItemExit.setMnemonic('x');
        menuItemExit.setText("Exit");
        menuItemExit.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuItemExitActionPerformed(evt);
            }
        });
        fileMenu.add(menuItemExit);

        jMenuBar1.add(fileMenu);

        goMenu.setMnemonic('g');
        goMenu.setText("Go");

        menuPreviousDay.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_LEFT, java.awt.event.InputEvent.ALT_MASK));
        menuPreviousDay.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/back.png"))); // NOI18N
        menuPreviousDay.setMnemonic('p');
        menuPreviousDay.setText("Previous Day");
        menuPreviousDay.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuPreviousDayActionPerformed(evt);
            }
        });
        goMenu.add(menuPreviousDay);

        menuToday.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_HOME, java.awt.event.InputEvent.ALT_MASK));
        menuToday.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/home.png"))); // NOI18N
        menuToday.setMnemonic('t');
        menuToday.setText("Today");
        menuToday.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuTodayActionPerformed(evt);
            }
        });
        goMenu.add(menuToday);

        menuNextDay.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_RIGHT, java.awt.event.InputEvent.ALT_MASK));
        menuNextDay.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/next.png"))); // NOI18N
        menuNextDay.setMnemonic('n');
        menuNextDay.setText("Next Day");
        menuNextDay.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuNextDayActionPerformed(evt);
            }
        });
        goMenu.add(menuNextDay);
        goMenu.add(jSeparator1);

        menuPreviousWeek.setText("Previous Week");
        menuPreviousWeek.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuPreviousWeekActionPerformed(evt);
            }
        });
        goMenu.add(menuPreviousWeek);

        menuNextWeek.setText("Next Week");
        menuNextWeek.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuNextWeekActionPerformed(evt);
            }
        });
        goMenu.add(menuNextWeek);
        goMenu.add(jSeparator2);

        menuPreviousMonth.setText("Previous Month");
        menuPreviousMonth.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuPreviousMonthActionPerformed(evt);
            }
        });
        goMenu.add(menuPreviousMonth);

        menuNextMonth.setText("Next Month");
        menuNextMonth.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuNextMonthActionPerformed(evt);
            }
        });
        goMenu.add(menuNextMonth);
        goMenu.add(jSeparator3);

        menuFirstUnread.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_U, java.awt.event.InputEvent.ALT_MASK));
        menuFirstUnread.setText("First Unread Item");
        menuFirstUnread.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                menuFirstUnreadActionPerformed(evt);
            }
        });
        goMenu.add(menuFirstUnread);

        jMenuBar1.add(goMenu);

        toolsMenu.setMnemonic('t');
        toolsMenu.setText("Tools");

        editorMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/edit.png"))); // NOI18N
        editorMenu.setText("Open Editor");
        editorMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                editorMenuActionPerformed(evt);
            }
        });
        toolsMenu.add(editorMenu);

        preferencesMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/process.png"))); // NOI18N
        preferencesMenu.setText("Preferences");
        preferencesMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                preferencesMenuActionPerformed(evt);
            }
        });
        toolsMenu.add(preferencesMenu);

        defineMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/dictionary.png"))); // NOI18N
        defineMenu.setText("jMenuItem1");
        defineMenu.setName("define"); // NOI18N
        defineMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                defineMenuActionPerformed(evt);
            }
        });
        toolsMenu.add(defineMenu);

        jMenuBar1.add(toolsMenu);

        helpMenu.setMnemonic('h');
        helpMenu.setText("Help");

        aboutMenu.setAccelerator(javax.swing.KeyStroke.getKeyStroke(java.awt.event.KeyEvent.VK_F1, 0));
        aboutMenu.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/help.png"))); // NOI18N
        aboutMenu.setText("About");
        aboutMenu.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                aboutMenuActionPerformed(evt);
            }
        });
        helpMenu.add(aboutMenu);

        jMenuBar1.add(helpMenu);

        setJMenuBar(jMenuBar1);
    }// </editor-fold>//GEN-END:initComponents

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing

	// WHAT TO DO?
	if (PropertyManager.getInstance().getPropertyAsBoolean(PropertyManager.PROPERTY_ALARM_CLOCK_MODE)) {
	    // HIDE, THEN START A THREAD TO UNHIDE AT THE CORRECT TIME
	    if (PropertyManager.getInstance().getProperty(PropertyManager.PROPERTY_WINDOW_HIDE_MODE).equals(PropertyManager.MODE_MINIMIZE)) {
		this.setExtendedState(ICONIFIED);
	    } else {
		this.setVisible(false);
	    }


	    // GET HOUR AND MINUTE TO WAKE UP
	    int h = 0;
	    int m = 0;
	    try {
		String time = PropertyManager.getInstance().getProperty(PropertyManager.PROPERTY_ALARM_CLOCK_TIME);
		StringTokenizer tok = new StringTokenizer(time, ".");
		String hour = tok.nextToken();
		String minute = tok.nextToken();
		h = Integer.parseInt(hour);
		m = Integer.parseInt(minute);
	    } catch (Exception e) {
		// IGNORE
	    }
	    Calendar cal = new GregorianCalendar();
	    cal.set(Calendar.DAY_OF_MONTH, (cal.get(Calendar.DAY_OF_MONTH) + 1));
	    cal.set(Calendar.HOUR_OF_DAY, h);
	    cal.set(Calendar.MINUTE, m);
	    this.timer.setNextEvent(cal.getTime());

	} else {

	    // save window position and size
	    PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_X, Integer.toString(this.getX()));
	    PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_Y, Integer.toString(this.getY()));
	    PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_WIDTH, Integer.toString(this.getWidth()));
	    PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_HEIGHT, Integer.toString(this.getHeight()));

	    // JUST EXIT
	    System.exit(0);
	}

    }//GEN-LAST:event_formWindowClosing


    public void savePositionAndSize() {
	// save window position and size
	PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_X, Integer.toString(this.getX()));
	PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_Y, Integer.toString(this.getY()));
	PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_WIDTH, Integer.toString(this.getWidth()));
	PropertyManager.getInstance().setProperty(PropertyManager.PROPERTY_WINDOW_HEIGHT, Integer.toString(this.getHeight()));

    }

    private void preferencesMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_preferencesMenuActionPerformed

	try {
	    new PreferencesDialog(this, true).setVisible(true);
	} catch (Throwable t) {
	    logger.error("ERROR", t);
	}

    }//GEN-LAST:event_preferencesMenuActionPerformed


    /**
     * Find the date with the first unread item and go to it.
     */
    private void menuFirstUnreadActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuFirstUnreadActionPerformed

	Date unreadItemDate = null;

	// GET SELECTED TAB
	Component c = this.tabPane.getSelectedComponent();
	if (c != null) {
	    if (c instanceof TabPanel) {
		logger.debug("Found tab panel.");
		try {
		    SDJEntryElement entry = ((TabPanel) c).getDataFile().getFirstUnreadEntry();
		    if (entry == null) {
			JOptionPane.showMessageDialog(this,
				"No unread items were found.",
				"Nothing Found",
				JOptionPane.INFORMATION_MESSAGE);
		    } else {
			logger.debug("Found " + entry + " as first unread item.");

			// ADJUST YEAR AS NEEDED
			String theDate = entry.getDate();
			if (((TabPanel) c).getDataFile().getSDJRootElement().isIgnoreYear()) {
			    String year = entryDateFormatter.format(new Date()).substring(0, 4);
			    theDate = year + entry.getDate().substring(4);
			}

			unreadItemDate = entryDateFormatter.parse(theDate);

			this.currentDate.setTime(unreadItemDate);
			updateDisplay();
		    }
		} catch (Exception e) {
		    logger.error("ERROR WHILE LOOKING FOR UNREAD ITEM.", e);
		    JOptionPane.showMessageDialog(this,
			    "An error occured while looking for the first\n" + "unread item (" + e.getMessage(),
			    "Error",
			    JOptionPane.ERROR_MESSAGE);
		}
	    } else {
		logger.debug("Not a tab panel.");
	    }
	} else {
	    logger.debug("No tab is selected.");
	}
    }//GEN-LAST:event_menuFirstUnreadActionPerformed


    /**
     * Display the main window after sleeping.
     *
     * <p>This kicks off the update checker, and sets the current day to
     * "today". It is called from the AlarmThread class.</p>
     */
    public void wakeup() {
	this.logger.debug("Time to wake up.");
	// CHECK FOR UPDATES
	this.updateButton.setVisible(false);
	if (PropertyManager.getInstance().getPropertyAsBoolean(PropertyManager.PROPERTY_CHECK_FOR_UPDATES)) {
	    // CHECK FOR UPDATES
	    Thread t = new Thread(new VersionChecker(this));
	    t.setDaemon(true);
	    t.start();
	}

	// GO TO TODAY
	this.today();

	// MAKE VISIBLE
	this.setExtendedState(NORMAL);
	this.setVisible(true);

    }


    /**
     * Do things that need to be done just before the window is displayed.
     * This method should be called by whatever creates the main window.
     * This stuff is here rather than in the constructor because we need to
     * make sure the class has been fully realized before doing some of it.
     */
    public void prepareDisplay() {
	int index = 0;
	// UPDATE THE CURRENT DAY DISPLAY
	this.createTabs();

	this.updateButton.setVisible(false);
	if (PropertyManager.getInstance().getPropertyAsBoolean(PropertyManager.PROPERTY_CHECK_FOR_UPDATES)) {
	    // CHECK FOR UPDATES
	    Thread t = new Thread(new VersionChecker(this));
	    t.setDaemon(true);
	    t.start();
	}
    }


    /**
     * Handle clicks on the update button.
     * The user will be asked if they want to go to the Scrip du Jour home page.
     * If so, we will try to open a browser.  If there is an error launching
     * the browser, let the user know, and tell them that they can download it
     * manually.
     */
	private void updateButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_updateButtonActionPerformed

	    String HOME_PAGE = "http://www.whirljack.net/software/scripdujour/";

	    int response = JOptionPane.showOptionDialog(this,
		    "A new version or Scrip du Jour is available!" + System.getProperty("line.separator") + "The new version can be downloaded from" + System.getProperty("line.separator") + HOME_PAGE + System.getProperty("line.separator") + System.getProperty("line.separator") + "Would you like to go there now?",
		    "New Version",
		    JOptionPane.YES_NO_OPTION,
		    JOptionPane.INFORMATION_MESSAGE,
		    null, null, null);

	    if (response == JOptionPane.YES_OPTION) {
		try {
		    BrowserLauncher.openURL(HOME_PAGE);
		} catch (Exception e) {
		    logger.error("Could not launch browser.", e);
		    JOptionPane.showMessageDialog(this,
			    "There was an error while launching a browser." + System.getProperty("line.separator") + "Point your browser to " + HOME_PAGE + System.getProperty("line.separator") + "to download the new version.",
			    "Error Launching Browser",
			    JOptionPane.INFORMATION_MESSAGE);
		}
	    }
	}//GEN-LAST:event_updateButtonActionPerformed


    /**
     * Navigate to the previous month.
     */
	private void menuPreviousMonthActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuPreviousMonthActionPerformed
	    this.previous(DateChangeMode.MONTH);
	}//GEN-LAST:event_menuPreviousMonthActionPerformed


    /**
     * Navigate to the next month.
     */
	private void menuNextMonthActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuNextMonthActionPerformed
	    this.next(DateChangeMode.MONTH);
	}//GEN-LAST:event_menuNextMonthActionPerformed


    /**
     * Navigate to the next week.
     */
	private void menuNextWeekActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuNextWeekActionPerformed
	    this.next(DateChangeMode.WEEK);
	}//GEN-LAST:event_menuNextWeekActionPerformed


    /**
     * Navigate to the previous week.
     */
	private void menuPreviousWeekActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuPreviousWeekActionPerformed
	    this.previous(DateChangeMode.WEEK);
	}//GEN-LAST:event_menuPreviousWeekActionPerformed


    /**
     * Display the editor.
     * This window is hidden, and a new EditorStartWindow is displayed.
     */
	private void editorMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_editorMenuActionPerformed

	    EditorStartWindow editorStartWindow = new EditorStartWindow();
	    this.setVisible(false);
	    editorStartWindow.setVisible(true);

	}//GEN-LAST:event_editorMenuActionPerformed


    /**
     * Allow the user to delete Scrip du Jour xml files.
     * A JFileChooser is displayed, allowing multiple selection, filtering
     * all *.xml files, and defaulting to the directory ~/.scripdujour, where
     * we store our xml files.
     * If the user selects some files and clicks "Delete Files", we try to delete
     * them, after asking nicely if they are sure.  If some files could not be
     * deleted, we tell the user.
     * After deleting files, the contents of the tabs are updated.
     */
    private void deleteMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_deleteMenuActionPerformed
	List<String> errList = new ArrayList<String>();

	JFileChooser jfc = new JFileChooser(SDJMain.getDataDir());
	jfc.setMultiSelectionEnabled(true);
	jfc.setFileFilter(new SwingXMLFileFilter());
	jfc.setApproveButtonText("Delete Files");
	jfc.setDialogTitle("Select Files to Delete");
	if (jfc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
	    File[] files = jfc.getSelectedFiles();
	    StringBuffer buf = new StringBuffer();
	    buf.append("You are about to delete the following files:" + System.getProperty("line.separator"));
	    for (File f : files) {
		buf.append(f.getAbsolutePath()).append(System.getProperty("line.separator"));
	    }
	    buf.append("Are you sure you want to do this?");

	    if (JOptionPane.showConfirmDialog(this, buf.toString(), "Delete Files?",
		    JOptionPane.YES_NO_OPTION,
		    JOptionPane.QUESTION_MESSAGE) == JOptionPane.OK_OPTION) {


		for (File f : files) {
		    if (!f.delete()) {
			logger.warn("Could not delete " + f.getAbsolutePath());
			errList.add(f.getAbsolutePath());
		    }
		}

		if (errList.size() > 0) {
		    buf.delete(0, buf.length());
		    buf.append("The following files could not be deleted:" + System.getProperty("line.separator"));
		    for (String s : errList) {
			buf.append(s).append(System.getProperty("line.separator"));
		    }
		    JOptionPane.showMessageDialog(this,
			    buf.toString(), "Error Deleting Files", JOptionPane.ERROR_MESSAGE);

		} else {
		    JOptionPane.showMessageDialog(this,
			    "Files deleted successfully.",
			    "Success", JOptionPane.INFORMATION_MESSAGE);
		}

		this.tabList = null;
		this.createTabs();
	    }
	}
    }//GEN-LAST:event_deleteMenuActionPerformed


    /**
     * Allow the user to install Scrip du Jour XML files.
     * A JFileChooser is displayed, with multiple selection enabled, defaulting
     * to the user home directory, and filtering all *.xml files.
     * If the user selects some files, they are copied to ~/.scripdujour,
     * and the tabs are updated.
     *
     * If some files could not be copied, the user is informed.
     */
    private void installMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_installMenuActionPerformed
	List<String> errList = new ArrayList<String>();
	List<String> successList = new ArrayList<String>();
	JFileChooser jfc = new JFileChooser(System.getProperty("user.home"));
	jfc.setApproveButtonText("Install Files");
	jfc.setMultiSelectionEnabled(true);
	jfc.setFileFilter(new SwingXMLFileFilter());

	if (jfc.showOpenDialog(this) == JFileChooser.APPROVE_OPTION) {
	    File[] files = jfc.getSelectedFiles();
	    for (File source : files) {
		File dest = new File(SDJMain.getDataDir(), source.getName());
		this.logger.debug("Copying " + source.getAbsolutePath() + " to " + dest.getAbsolutePath());
		try {
		    FileUtil.copy(source, dest);
		    successList.add(source.getName());
		} catch (IOException ioe) {
		    errList.add(source.getAbsolutePath());
		}
	    }
	    if (errList.size() > 0) {
		StringBuilder sb = new StringBuilder();
		sb.append("The following files could not be copied:" + System.getProperty("line.separator"));
		for (String s : errList) {
		    sb.append(s).append(System.getProperty("line.separator"));
		}
		JOptionPane.showMessageDialog(this,
			sb.toString(), "Error Copying Files", JOptionPane.ERROR_MESSAGE);
	    } else {
		StringBuilder sb = new StringBuilder();
		sb.append("The following files have been copied into Scrip du Jour's data directory:");
		sb.append(System.getProperty("line.separator"));
		for (String s : successList) {
		    sb.append(s).append(System.getProperty("line.separator"));
		}
		sb.append("You can delete these files if you wish.");

		JOptionPane.showMessageDialog(this,
			sb.toString(),
			"Files Copied Successfully",
			JOptionPane.INFORMATION_MESSAGE);
	    }

	    this.tabList = null;
	    this.createTabs();
	}
    }//GEN-LAST:event_installMenuActionPerformed


    /**
     * Display the about dialog as modal.
     */
	private void aboutMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_aboutMenuActionPerformed
	    new AboutDialog(this, true).setVisible(true);
	}//GEN-LAST:event_aboutMenuActionPerformed


    /**
     * When a user selects the next menu, go to the next day.
     */
	private void menuNextDayActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuNextDayActionPerformed
	    this.next(DateChangeMode.DAY);
	}//GEN-LAST:event_menuNextDayActionPerformed


    /**
     * When a user selects the home menu, go to today.
     */
	private void menuTodayActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuTodayActionPerformed
	    this.today();
	}//GEN-LAST:event_menuTodayActionPerformed


    /**
     * When a user selects the back menu, go to the previous day.
     */
	private void menuPreviousDayActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuPreviousDayActionPerformed
	    this.previous(DateChangeMode.DAY);
	}//GEN-LAST:event_menuPreviousDayActionPerformed


    /**
     * When the user clicks the home button, go to today.
     */
	private void homeButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_homeButtonActionPerformed
	    this.today();
	}//GEN-LAST:event_homeButtonActionPerformed


    /**
     * When the user clicks the next button, go to the next day.
     */
	private void nextButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_nextButtonActionPerformed
	    this.next(DateChangeMode.DAY);
	}//GEN-LAST:event_nextButtonActionPerformed


    /**
     * When the user clicks the back button, go to the previous day.
     */
	private void backButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_backButtonActionPerformed
	    this.previous(DateChangeMode.DAY);
	}//GEN-LAST:event_backButtonActionPerformed


    /**
     * Set the date to today.
     */
    void today() {
	this.currentDate.setTime(new Date());
	updateDisplay();
    }


    /**
     * Navigate to the next day, week, or month.
     * @param changeBy the unit of time to change the date by.
     *
     */
    void next(DateChangeMode changeBy) {
	switch (changeBy) {
	    case DAY:
		this.currentDate.add(GregorianCalendar.DAY_OF_MONTH, 1);
		break;

	    case WEEK:
		this.currentDate.add(GregorianCalendar.WEEK_OF_YEAR, 1);
		break;

	    case MONTH:
		this.currentDate.add(GregorianCalendar.MONTH, 1);
		break;

	    default:
		logger.warn("INVALID CHANGE MODE: " + changeBy);
		break;
	}
	updateDisplay();

    }


    /**
     * Navigate to the previous day, week, or month.
     * @param changeBy the unit of time to change the date by.
     *
     */
    void previous(DateChangeMode changeBy) {
	switch (changeBy) {
	    case DAY:
		this.currentDate.add(GregorianCalendar.DAY_OF_MONTH, -1);
		break;

	    case WEEK:
		this.currentDate.add(GregorianCalendar.WEEK_OF_YEAR, -1);
		break;

	    case MONTH:
		this.currentDate.add(GregorianCalendar.MONTH, -1);
		break;

	    default:
		logger.warn("INVALID CHANGE MODE: " + changeBy);
		break;
	}
	updateDisplay();
    }


    /**
     * Called when a new version is available.
     */
    public void newVersionAvailable() {
	this.updateButton.setVisible(true);
    }


    protected List<TabPanel> getTabList() {
	return this.tabList;
    }


    /**
     * Goodbye.
     */
	private void menuItemExitActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_menuItemExitActionPerformed
	    System.exit(0);
	}//GEN-LAST:event_menuItemExitActionPerformed


    /**
     * Display definition for the selected word.
     * @param evt
     */
	private void defineMenuActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_defineMenuActionPerformed
	    TabPanel tp = (TabPanel) this.tabPane.getSelectedComponent();
	    DefinitionDialog def = new DefinitionDialog(SDJMain.getMainWindow(), false);
	    def.setVisible(true);
	    def.define(tp.getWordToDefine());
	}//GEN-LAST:event_defineMenuActionPerformed


    /**
     * Update the display when the user navigates to a new day.
     * The work is performed by an instance of RecreateTabs, and the work
     * is executed in a new thread using the SwingWorker.
     */
    private void updateDisplay() {
	// CHANGE THE DISPLAYED DATE
	this.lblDate.setText("Showing data for " + this.dateFormatter.format(this.currentDate.getTime()));

	// UPDATE THE CONTENT OF THE TABS IN A SEPARATE THREAD,
	// DISPLAYING A WAIT DIALOG
	setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	final WaitDialog wait = new WaitDialog(SDJMain.getMainWindow(), true, "Updating display...");
	SwingWorker worker = new SwingWorker() {

	    public Object construct() {
		return new UpdateDisplay();
	    }


	    @Override
	    public void finished() {
		// MAKE SURE THE WAIT DIALOG IS VISIBLE
		// BEFORE HIDING IT
		while (!wait.isVisible()) {
		    try {
			Thread.sleep(500);
		    } catch (Exception e) {
			logger.warn("Interrupted.", e);
		    }
		}
		wait.setVisible(false);
		wait.dispose();
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	    }

	};
	worker.start();
	wait.setVisible(true);
    }


    /**
     * Re-create the tabs when a user adds or deletes a data file.
     * The work is performed by an instance of RecreateTabs, and the work
     * is executed in a new thread using the SwingWorker.
     */
    private void createTabs() {
	// CHANGE THE DISPLAYED DATE
	this.lblDate.setText("Showing data for " + this.dateFormatter.format(this.currentDate.getTime()));

	// UPDATE THE CONTENT OF THE TABS IN A SEPARATE THREAD,
	// DISPLAYING A WAIT DIALOG
	setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

	final WaitDialog wait = new WaitDialog(SDJMain.getMainWindow(), true, "Updating display...");

	SwingWorker worker = new SwingWorker() {

	    public Object construct() {
		return new CreateTabs();
	    }


	    @Override
	    public void finished() {
		// MAKE SURE THE WAIT DIALOG IS VISIBLE
		// BEFORE HIDING IT
		while (!wait.isVisible()) {
		    try {
			Thread.sleep(500);
		    } catch (Exception e) {
			logger.warn("Interrupted.", e);
		    }
		}
		wait.setVisible(false);
		wait.dispose();
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	    }

	};
	worker.start();
	wait.setVisible(true);

    }


    /**
     * Create the tabs.
     * If the tab list is null, the TabBuilder will be called to get a
     * list, and the list will be set.
     */
    class CreateTabs {

	CreateTabs() {
	    if (tabList == null) {
		tabList = TabBuilder.getInstance().createTabs();
	    }

	    tabPane.removeAll();
	    if (tabList.size() > 0) {
		int index = 0;
		for (TabPanel tab : tabList) {
		    tab.displayDataForDate(currentDate.getTime());
		    tabPane.add(tab.getTabTitle(), tab);

		    // tell the tab panel which index it is using
		    tab.setIndex(index);
		    tab.setTabbedPane(tabPane);
		    index++;
		}
	    } else {
		javax.swing.JTextArea t = new javax.swing.JTextArea();
		t.setText("Welcome to Scrip du Jour!" + System.getProperty("line.separator") + System.getProperty("line.separator") + "The program is installed and ready to display things for you to read, " + "but there are no data files installed.  Use the \"Install Data Files\" option " + "in the \"File\" menu to locate and install data files.");
		t.setWrapStyleWord(true);
		t.setEditable(false);
		t.setLineWrap(true);

		tabPane.add(t);
	    }
	}

    }


    /**
     * Update the display, telling each tab to display data for the current date.
     * This is executed in a SwingWorker so we can display a modal wait dialog
     * while the tabs are looking up data.
     */
    class UpdateDisplay {

	UpdateDisplay() {
	    Date date = currentDate.getTime();
	    for (TabPanel tab : tabList) {
		tab.displayDataForDate(date);
		tab.updateTabTitle();
	    }
	}

    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuItem aboutMenu;
    private javax.swing.JButton backButton;
    private javax.swing.JMenuItem defineMenu;
    private javax.swing.JMenuItem deleteMenu;
    private javax.swing.JMenuItem editorMenu;
    private javax.swing.JMenu fileMenu;
    private javax.swing.JMenu goMenu;
    private javax.swing.JMenu helpMenu;
    private javax.swing.JButton homeButton;
    private javax.swing.JMenuItem installMenu;
    private javax.swing.JMenuBar jMenuBar1;
    private javax.swing.JSeparator jSeparator1;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JToolBar jToolBar1;
    private javax.swing.JLabel lblDate;
    private javax.swing.JMenuItem menuFirstUnread;
    private javax.swing.JMenuItem menuItemExit;
    private javax.swing.JMenuItem menuNextDay;
    private javax.swing.JMenuItem menuNextMonth;
    private javax.swing.JMenuItem menuNextWeek;
    private javax.swing.JMenuItem menuPreviousDay;
    private javax.swing.JMenuItem menuPreviousMonth;
    private javax.swing.JMenuItem menuPreviousWeek;
    private javax.swing.JMenuItem menuToday;
    private javax.swing.JButton nextButton;
    private javax.swing.JMenuItem preferencesMenu;
    private javax.swing.JTabbedPane tabPane;
    private javax.swing.JMenu toolsMenu;
    private javax.swing.JButton updateButton;
    // End of variables declaration//GEN-END:variables


    class ToolsMenuHandler implements MenuListener {

	/**
	 * When the Tools menu is clicked, check for a selection in the
	 * tab panel and set the "Define" menu item accordingly.
	 *
	 * @param e
	 */
	@Override
	public void menuSelected(MenuEvent e) {
	    // is there something selected in the current tab?
	    Component c = tabPane.getSelectedComponent();

	    if (c instanceof TabPanel) {
		TabPanel tp = (TabPanel) c;
		String word = tp.getWordToDefine();
		JMenu m = (JMenu) e.getSource();
		for (Component menu : m.getMenuComponents()) {
		    if (menu instanceof JMenuItem) {
			JMenuItem jmenu = (JMenuItem) menu;
			String name = jmenu.getName();
			if (name != null && name.equals("define")) {
			    if (word == null) {
				jmenu.setText("Define");
				jmenu.setEnabled(false);
			    } else {
				jmenu.setText("Define \"" + word + "\"");
				jmenu.setEnabled(true);
			    }
			}
		    }
		}
	    }
	}


	@Override
	public void menuDeselected(MenuEvent e) {
	}


	@Override
	public void menuCanceled(MenuEvent e) {
	}

    }
}
