/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package net.whirljack.sdj.gui;

// JAVA I/O
import java.io.File;

// JAVA UTILITY
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

// SWING
import javax.swing.JOptionPane;

// SDJ
import net.whirljack.sdj.SDJMain;
import net.whirljack.sdj.XMLParser;
import net.whirljack.sdj.bo.SDJRootElement;

// LOGGING
import org.apache.log4j.Logger;



/**
 * Build tabs from the Scrip du Jour XML files.
 *
 * @author jeremyb
 */
public class TabBuilder {
    
    /** Logging. */
    private Logger logger = Logger.getLogger(TabBuilder.class);
    
    /** Reference to the only instance of this class. */
    private static TabBuilder instance = null;
    
    /** Filter xml files. */
    private java.io.FileFilter fileFilter = new XMLFileFilter();
    
    /**
     * Creates a new instance of TabBuilder
     */
    private TabBuilder() {
    }
    
    
    /**
     * Get reference to this class.
     * @return reference to TabBuilder instance.
     */
    public static TabBuilder getInstance() {
	if (instance == null) {
	    instance = new TabBuilder();
	}
	
	return instance;
    }
    
    
    /**
     * Data files should be located in ~/.scripdujour.
     * Any xml file will be parsed, but only files with the correct
     * structure will be used.
     *
     * The required structure is:
     *
     * <document description="" shortDescription="" ignoreYear="">
     *   <entry date="yyyymmdd" read="">
     *     <heading></heading>
     *     <text></text>
     *   </entry>
     * </document>
     *
     * The description field will appear inside the tab.  The short description
     * will appear on the tab.  Heading and text are displayed per date, per tab.
     * There can be multiple entry elements.
     *
     * An error is displayed if there is a problem with some files.
     *
     * @return 
     */
    public List<TabPanel> createTabs() {
	File[] files = SDJMain.getDataDir().listFiles(this.fileFilter);
	if (files != null) {
	    Arrays.sort(files);
	    logger.debug("Got " + files.length + " files.");
	}
	
	List<TabPanel> tabList = new LinkedList<TabPanel>();
	SDJRootElement root = null;
	
	for (File file : files) {
	    try {
		String name = file.getName();
		if (name.endsWith(".xml")) {
		    logger.debug("Parsing file " + file.getAbsolutePath());
		    root = XMLParser.getInstance().parseRootElement(file);
		    if (root != null) {
			tabList.add(new TabPanel(root));
		    } else {
			this.logger.warn("XML file " + file.getAbsolutePath()
			+ " does not appear to be a Scrip du Jour file.");
		    }
		}
	    } catch (Exception e) {
		// HIDE SPLASH WINDOW; IF IT IS SHOWING, THE ERROR DIALOG
		// WILL NOT BE VISIBLE
		SplashWindow.getInstance().close();
		
		logger.error("ERROR WHILE PARSING FILE.", e);
		JOptionPane.showMessageDialog(null, "There was an error while parsing file '" + file.getName()
		+ System.getProperty("line.separator")
		+ "I will attempt to continue, but the data in the bad file"
			+ System.getProperty("line.separator")
			+ "will not be used.  Perhaps the file was corrupted?"
			+ System.getProperty("line.separator")
			+ System.getProperty("line.separator")
			+ "Hint: You can use Tools -> Delete Data Files to"
			+ System.getProperty("line.separator")
			+ "delete problem files.",
			"Parse Error", JOptionPane.WARNING_MESSAGE);
	    }
	}
	
	return tabList;
    }
    
    
    /**
     * Filter xml files.
     */
    private class XMLFileFilter implements java.io.FileFilter {
	public boolean accept(File pathname) {
	    boolean ok = false;
	    
	    if (pathname != null && pathname.getName().toLowerCase().endsWith(".xml")) {
		ok = true;
	    }
	    
	    return ok;
	}
	
    }
    
}
