/* Copyright 2006, Jeremy Brooks <jeremyb@whirljack.net>
 *
 * This file is part of Scrip du Jour.
 *
 * Scrip du Jour is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * Scrip du Jour is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package net.whirljack.sdj.gui;

// JAVA AWT
import java.awt.Component;
import java.awt.Cursor;

// JAVA TEXT
import java.awt.event.MouseEvent;
import java.text.SimpleDateFormat;

// JAVA UTILITY
import java.util.Date;

// JAVA SWING
import java.util.List;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;

// SDJ
import javax.swing.JTextArea;
import net.jeremybrooks.knicker.Knicker;
import net.jeremybrooks.knicker.dto.Definition;
import net.whirljack.sdj.PropertyManager;
import net.whirljack.sdj.SDJMain;
import net.whirljack.sdj.bo.SDJDataFile;
import net.whirljack.sdj.bo.SDJRootElement;
import net.whirljack.sdj.bo.SDJEntryElement;

// LOGGING
import org.apache.log4j.Logger;


/**
 * A panel representing data for a SDJMain XML file.
 *
 * @author  jeremyb
 */
public class TabPanel extends javax.swing.JPanel {

    /** Logging. */
    private Logger logger = Logger.getLogger(TabPanel.class);

    /** The root element of the file this panel represents. */
    private SDJRootElement root;

    /** The data structure backing the display. */
    private SDJDataFile dataFile;

    /** Format dates to/from YYYYMMDD. */
    private SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyyMMdd");

    /** The current date displayed by this panel. */
    private Date currentDate = null;

    /** The index of this tab in the tab pane. */
    private int index;

    /** The tab pane that contains this tab panel. */
    private JTabbedPane tabbedPane;


    /**
     * Default constructor is private to force use of other constructors.
     */
    private TabPanel() {
    }


    /**
     * Create a TabPanel object representing the specified root element.
     * @param root the root element from the file we are showing.
     * @throws Exception if the data file cannot be created.
     */
    public TabPanel(SDJRootElement root) throws Exception {
	this.root = root;
	this.dataFile = new SDJDataFile(root.getFile());
	initComponents();
    }


    /**
     * Get the short description from the root element.
     *
     * @return short description attribute.
     */
    public String getShortDescription() {
	return this.root.getShortDescription();
    }


    /**
     * Get the tab title.
     * The tab title is the short description with the unread item count.
     *
     * @return tab title.
     */
    public String getTabTitle() {
	int unreadItemCount = this.dataFile.getUnreadItemCount();
	if (unreadItemCount == 0) {
	    return this.getShortDescription();
	} else {
	    return this.getShortDescription() + " [" + unreadItemCount + "]";
	}
    }


    /**
     * Update the title.
     */
    public void updateTabTitle() {
	this.tabbedPane.setTitleAt(this.index, this.getTabTitle());
    }


    /**
     * Set the index.
     * 
     * @param index
     */
    public void setIndex(int index) {
	this.index = index;
    }


    /**
     * Get the index for this tab.
     *
     * @return index of this tab in the tab pane.
     */
    public int getIndex() {
	return this.index;
    }


    /**
     * Set the tabbed pane that contains this tab panel.
     * 
     * @param tabbedPane the JTabbedPane that contains this tab panel.
     */
    public void setTabbedPane(JTabbedPane tabbedPane) {
	this.tabbedPane = tabbedPane;
    }


    /**
     * Get the tabbed pae that contains this tab panel.
     *
     * @return the JTabbedPane instance that contains this tab panel.
     */
    public JTabbedPane getTabbedPane() {
	return this.tabbedPane;
    }


    public void updateFontSize(float size) {
	txtHeading.setFont(txtHeading.getFont().deriveFont(size));
	txtText.setFont(txtText.getFont().deriveFont(size));
    }


    /**
     * Display data for a specific date.
     *
     * @param date the date to display data for.
     */
    public void displayDataForDate(Date date) {

	this.logger.debug("Trying to find entry for date " + date.toString() + " - ignore year is " + this.root.isIgnoreYear());
	this.currentDate = date;

	try {
	    SDJEntryElement entry = this.dataFile.getEntryForDate(date);
	    if (entry == null) {
		this.txtHeading.setText("No entry found for date.");
		this.txtText.setText("");
		this.cbxRead.setSelected(false);
		this.cbxRead.setEnabled(false);
	    } else {
		this.txtHeading.setText(entry.getHeading());
		this.txtText.setText(entry.getText());
		this.cbxRead.setSelected(entry.isRead());
		this.cbxRead.setEnabled(true);
	    }

	    // THIS FORCES THE VIEWPORT TO SCROLL TO THE TOP OF THE NEWLY ADDED TEXT
	    // setCaretPosition() DID NOT SEEM TO WORK, THIS DOES.
	    this.txtText.setSelectionStart(1);


	} catch (Exception e) {
	    this.logger.error("Error getting entry for date " + date, e);
	}
    }


    /**
     * This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        mnuPopup = new javax.swing.JPopupMenu();
        mnuPrevious = new javax.swing.JMenuItem();
        mnuToday = new javax.swing.JMenuItem();
        mnuNext = new javax.swing.JMenuItem();
        jSeparator1 = new javax.swing.JPopupMenu.Separator();
        mnuRead = new javax.swing.JCheckBoxMenuItem();
        jSeparator2 = new javax.swing.JPopupMenu.Separator();
        mnuDefine = new javax.swing.JMenuItem();
        lblDescription = new javax.swing.JLabel();
        txtHeading = new javax.swing.JTextArea();
        jScrollPane1 = new javax.swing.JScrollPane();
        txtText = new javax.swing.JTextArea();
        cbxRead = new javax.swing.JCheckBox();

        mnuPrevious.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/back.png"))); // NOI18N
        mnuPrevious.setText("Previous Day");
        mnuPrevious.setToolTipText("Go to the previous day.");
        mnuPrevious.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                mnuPreviousActionPerformed(evt);
            }
        });
        mnuPopup.add(mnuPrevious);

        mnuToday.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/home.png"))); // NOI18N
        mnuToday.setText("Today");
        mnuToday.setToolTipText("Go to today.");
        mnuToday.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                mnuTodayActionPerformed(evt);
            }
        });
        mnuPopup.add(mnuToday);

        mnuNext.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/next.png"))); // NOI18N
        mnuNext.setText("Next Day");
        mnuNext.setToolTipText("Go to the next day.");
        mnuNext.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                mnuNextActionPerformed(evt);
            }
        });
        mnuPopup.add(mnuNext);
        mnuPopup.add(jSeparator1);

        mnuRead.setSelected(true);
        mnuRead.setText("I've read it");
        mnuRead.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                mnuReadActionPerformed(evt);
            }
        });
        mnuPopup.add(mnuRead);
        mnuPopup.add(jSeparator2);

        mnuDefine.setIcon(new javax.swing.ImageIcon(getClass().getResource("/net/whirljack/sdj/images/dictionary.png"))); // NOI18N
        mnuDefine.setText("Define");
        mnuDefine.setToolTipText("Define the selected word.");
        mnuDefine.setName("define"); // NOI18N
        mnuDefine.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                mnuDefineActionPerformed(evt);
            }
        });
        mnuPopup.add(mnuDefine);

        setLayout(new java.awt.GridBagLayout());

        lblDescription.setFont(new java.awt.Font("SansSerif", 3, 12));
        lblDescription.setText(this.root.getDescription());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        add(lblDescription, gridBagConstraints);

        txtHeading.setEditable(false);
        txtHeading.setFont(txtHeading.getFont().deriveFont(PropertyManager.getInstance().getPropertyAsFloat(PropertyManager.PROPERTY_FONT_SIZE)));
        txtHeading.setLineWrap(true);
        txtHeading.setWrapStyleWord(true);
        txtHeading.setMargin(new java.awt.Insets(3, 3, 3, 3));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        add(txtHeading, gridBagConstraints);

        jScrollPane1.setAutoscrolls(true);

        txtText.setEditable(false);
        txtText.setFont(txtText.getFont().deriveFont(PropertyManager.getInstance().getPropertyAsFloat(PropertyManager.PROPERTY_FONT_SIZE)));
        txtText.setLineWrap(true);
        txtText.setWrapStyleWord(true);
        txtText.setMargin(new java.awt.Insets(3, 3, 3, 3));
        txtText.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                txtTextMousePressed(evt);
            }
            public void mouseReleased(java.awt.event.MouseEvent evt) {
                txtTextMouseReleased(evt);
            }
        });
        jScrollPane1.setViewportView(txtText);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        add(jScrollPane1, gridBagConstraints);

        cbxRead.setText("I've read it");
        cbxRead.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cbxReadActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        add(cbxRead, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents


    /**
     * Handle when a user clicks the "I've read it" checkbox.
     *
     * <p>This action creates a new instance of UpdateReadFlag, executing that
     * in a SwingWorker.</p>
     *
     */
	private void cbxReadActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cbxReadActionPerformed

	    setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
	    final WaitDialog wait = new WaitDialog(SDJMain.getMainWindow(), true, "Updating file...");
	    SwingWorker worker = new SwingWorker() {

		public Object construct() {
		    return new UpdateReadFlag();
		}


		@Override
		public void finished() {
		    // MAKE SURE THE WAIT DIALOG IS VISIBLE
		    // BEFORE HIDING IT
		    while (!wait.isVisible()) {
			try {
			    Thread.sleep(500);
			} catch (Exception e) {
			    logger.warn("Interrupted.", e);
			}
		    }
		    wait.setVisible(false);
		    wait.dispose();
		    setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		}

	    };
	    worker.start();

	    wait.setVisible(true);
	}//GEN-LAST:event_cbxReadActionPerformed

	private void txtTextMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_txtTextMousePressed
	    if (evt.isPopupTrigger()) {
		this.showPopup(evt);
	    } else {
		if (evt.getClickCount() == 2) {
		    // nothing to do on a double click for now
		}
	    }
	}//GEN-LAST:event_txtTextMousePressed


    public void showPopup(MouseEvent e) {

	if (e.getComponent() instanceof JTextArea) {
	    // set the state of the checkbox and define menu
	    for (Component c : this.mnuPopup.getComponents()) {
		if (c instanceof JCheckBoxMenuItem) {
		    ((JCheckBoxMenuItem) c).setSelected(this.cbxRead.isSelected());
		} else if (c instanceof JMenuItem) {
		    JMenuItem menu = (JMenuItem) c;
		    String name = menu.getName();
		    if (name != null && name.equals("define")) {
			if (this.getWordToDefine() == null) {
			    menu.setText("Define");
			    menu.setEnabled(false);
			} else {
			    menu.setText("Define \"" + this.getWordToDefine() + "\"");
			    menu.setEnabled(true);
			}
		    }
		}
	    }

	    this.mnuPopup.show(e.getComponent(), e.getX(), e.getY());
	}
    }


    /**
     * Get the selected word.
     *
     * <p>The user must have only one word selected. If there are spaces in the
     * selection, it is considered more than one word and will return null.
     * Punctuation and numbers are ignored and will not be returned as part of
     * the word.</p>
     *
     * @return the selected word, or null if a valid selection is not found.
     */
    public String getWordToDefine() {
	String selection = this.txtText.getSelectedText();
	String retString = null;
	if (selection != null && !selection.isEmpty()) {

	    // only continue if there are no spaces in the selection
	    if (!selection.contains(" ")) {

		// get rid of all characters that are not letters
		StringBuilder sb = new StringBuilder();
		for (char c : selection.toCharArray()) {
		    if (Character.isLetter(c)) {
			sb.append(c);
		    }
		}

		// if there are still characters, we have a word to define
		if (sb.length() > 0) {
		    retString = sb.toString();
		}
	    }
	}

	return retString;
    }


	private void mnuNextActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mnuNextActionPerformed
	    SDJMain.getMainWindow().next(MainWindow.DateChangeMode.DAY);
	}//GEN-LAST:event_mnuNextActionPerformed

	private void mnuTodayActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mnuTodayActionPerformed
	    SDJMain.getMainWindow().today();
	}//GEN-LAST:event_mnuTodayActionPerformed

	private void mnuPreviousActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mnuPreviousActionPerformed
	    SDJMain.getMainWindow().previous(MainWindow.DateChangeMode.DAY);
	}//GEN-LAST:event_mnuPreviousActionPerformed

	private void mnuReadActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mnuReadActionPerformed
	    // toggle the state of the checkbox
	    this.cbxRead.setSelected(!cbxRead.isSelected());

	    // then trigger the action
	    this.cbxReadActionPerformed(null);
	}//GEN-LAST:event_mnuReadActionPerformed

	private void mnuDefineActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_mnuDefineActionPerformed
	    DefinitionDialog def = new DefinitionDialog(SDJMain.getMainWindow(), false);
	    def.setVisible(true);
	    def.define(this.getWordToDefine());
	}//GEN-LAST:event_mnuDefineActionPerformed

	private void txtTextMouseReleased(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_txtTextMouseReleased
	    if (evt.isPopupTrigger()) {
		this.showPopup(evt);
	    } else {
		if (evt.getClickCount() == 2) {
		    // nothing to do on a double click for now
		}
	    }
	}//GEN-LAST:event_txtTextMouseReleased


    /**
     * Get the data file backing this tab.
     * @return instance of the data file.
     */
    public SDJDataFile getSdjDataFile() {
	return this.dataFile;
    }


    /**
     * Get the number of unread items in the data for this tab.
     *
     * @return number of unread items.
     */
    public int getUnreadItemCount() {
	return this.dataFile.getUnreadItemCount();
    }


    /**
     * Get a reference to the data file for this tab.
     *
     * @return reference to the data file instance for this tab.
     */
    public SDJDataFile getDataFile() {
	return this.dataFile;
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JCheckBox cbxRead;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JPopupMenu.Separator jSeparator1;
    private javax.swing.JPopupMenu.Separator jSeparator2;
    private javax.swing.JLabel lblDescription;
    private javax.swing.JMenuItem mnuDefine;
    private javax.swing.JMenuItem mnuNext;
    private javax.swing.JPopupMenu mnuPopup;
    private javax.swing.JMenuItem mnuPrevious;
    private javax.swing.JCheckBoxMenuItem mnuRead;
    private javax.swing.JMenuItem mnuToday;
    private javax.swing.JTextArea txtHeading;
    private javax.swing.JTextArea txtText;
    // End of variables declaration//GEN-END:variables


    /**
     * This is the work done when a user clicks on the
     * "I've read it" checkbox.  This work is executed by a SwingWorker.
     *
     */
    class UpdateReadFlag {

	UpdateReadFlag() {
	    dataFile.setReadFlag(currentDate, cbxRead.isSelected());
	    try {
		dataFile.write();
		updateTabTitle();
	    } catch (Exception e) {
		logger.error("ERROR SAVING FILE.", e);
		JOptionPane.showMessageDialog(null,
			"There was an error while saving the file.\n" + "Error message: " + e.toString(),
			"Save Error",
			JOptionPane.ERROR_MESSAGE);
	    }
	}

    }
}
